/******************************************************************************
*                                                                             *
*    Extract.c                              Copyright(c) 2005-2010 itow,y.    *
*                                                                             *
******************************************************************************/

/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <windows.h>
#include <tchar.h>
#include "DataCutter.h"
#include "Extract.h"
#include "Formats.h"


#define PROGRESS_BYTES 1024

#define MIN_DIB_FILE_SIZE (sizeof(BITMAPFILEHEADER)+sizeof(BITMAPINFOHEADER)+4)
#define MIN_DIB_SIZE (sizeof(BITMAPINFOHEADER)+4)
#define MIN_WAVE_FILE_SIZE (8+4+8+(sizeof(WAVEFORMATEX)-2)+8+2)
#define MIN_FILE_SIZE min(MIN_DIB_SIZE,MIN_WAVE_FILE_SIZE)

#define MAX_DIB_SIZE 32767

#define MIN_MP3_FRAME_SIZE 96
#define MAX_MP3_FRAME_SIZE (1440+1)




static DataSizeType ExtractDIBFileData(const BYTE *pData,DataSizeType DataSize,
							ExtractDataCallback pfnDataCallback,void *pParam)
{
	const BITMAPFILEHEADER *pbmfh;
	const BITMAPINFOHEADER *pbmih;
	DataSizeType Size;

	pbmfh=(const BITMAPFILEHEADER*)pData;
	if (pbmfh->bfSize<MIN_DIB_FILE_SIZE || pbmfh->bfSize>DataSize
			|| pbmfh->bfReserved1!=0 || pbmfh->bfReserved2!=0
			|| pbmfh->bfOffBits<
						sizeof(BITMAPFILEHEADER)+sizeof(BITMAPINFOHEADER))
		return 1;
	pbmih=(const BITMAPINFOHEADER*)(pData+sizeof(BITMAPFILEHEADER));
	if (pbmih->biSize!=sizeof(BITMAPINFOHEADER)
			|| pbmih->biWidth<=0 || pbmih->biHeight==0
			|| pbmih->biWidth>MAX_DIB_SIZE || abs(pbmih->biHeight)>MAX_DIB_SIZE
			|| pbmih->biPlanes!=1
			|| (pbmih->biBitCount!=1 && pbmih->biBitCount!=4
				&& pbmih->biBitCount!=8 && pbmih->biBitCount!=16
				&& pbmih->biBitCount!=24 && pbmih->biBitCount!=32)
			|| (pbmih->biCompression!=BI_RGB
				&& pbmih->biCompression!=BI_RLE4
				&& pbmih->biCompression!=BI_RLE8
				&& pbmih->biCompression!=BI_BITFIELDS)
			|| (pbmih->biCompression==BI_RLE4 && pbmih->biBitCount!=4)
			|| (pbmih->biCompression==BI_RLE8 && pbmih->biBitCount!=8)
			|| (pbmih->biCompression==BI_BITFIELDS
				&& (pbmih->biBitCount!=16 && pbmih->biBitCount!=32))
			|| (pbmih->biSizeImage==0
				&& (pbmih->biCompression!=BI_RLE4
					|| pbmih->biCompression!=BI_RLE8))
			|| (pbmih->biSizeImage!=0
				&& (pbmih->biCompression==BI_RGB
					|| pbmih->biCompression==BI_BITFIELDS)
				&& pbmih->biSizeImage!=
							DIB_ROW_BYTES(pbmih->biWidth,pbmih->biBitCount)*
															pbmih->biHeight))
		return sizeof(BITMAPFILEHEADER);
	Size=sizeof(BITMAPFILEHEADER)+sizeof(BITMAPINFOHEADER);
	if (pbmih->biBitCount<=8) {
		if (pbmih->biClrUsed==0
				|| pbmih->biClrUsed==(DWORD)(1<<pbmih->biBitCount)
				|| pbmfh->bfOffBits-sizeof(BITMAPFILEHEADER)-
													sizeof(BITMAPINFOHEADER)==
										(1<<pbmih->biBitCount)*sizeof(RGBQUAD))
			Size+=(DataSizeType)(1<<pbmih->biBitCount)*sizeof(RGBQUAD);
		else if (pbmfh->bfOffBits-sizeof(BITMAPFILEHEADER)-
					sizeof(BITMAPINFOHEADER)==pbmih->biClrUsed*sizeof(RGBQUAD))
			/* biClrUsed ̐pbg̐F݂Ȃt@C */
			Size+=pbmih->biClrUsed*sizeof(RGBQUAD);
		else
			return sizeof(BITMAPFILEHEADER)+sizeof(BITMAPINFOHEADER);
	}
	if (pbmih->biCompression==BI_RLE4 || pbmih->biCompression==BI_RLE8) {
		Size+=pbmih->biSizeImage;
	} else {
		if (pbmih->biCompression==BI_BITFIELDS)
			Size+=3*sizeof(DWORD);
		Size+=DIB_ROW_BYTES(pbmih->biWidth,pbmih->biBitCount)*pbmih->biHeight;
	}
	if (Size>DataSize)
		return sizeof(BITMAPFILEHEADER)+sizeof(BITMAPINFOHEADER);
	if (!(*pfnDataCallback)(FORMAT_DIB,pData,Size,pParam))
		return 0;
	return Size;
}


static DataSizeType ExtractDIBData(const BYTE *pData,DataSizeType DataSize,
							ExtractDataCallback pfnDataCallback,void *pParam)
{
	const BITMAPINFOHEADER *pbmih;
	DataSizeType Size,PixelBytes;
	BOOL fIcon=FALSE;

	pbmih=(const BITMAPINFOHEADER*)pData;
	if (pbmih->biWidth<=0 || pbmih->biHeight==0
			|| pbmih->biWidth>MAX_DIB_SIZE || abs(pbmih->biHeight)>MAX_DIB_SIZE
			|| pbmih->biPlanes!=1
			|| (pbmih->biBitCount!=1 && pbmih->biBitCount!=4
				&& pbmih->biBitCount!=8 && pbmih->biBitCount!=16
				&& pbmih->biBitCount!=24 && pbmih->biBitCount!=32)
			|| (pbmih->biCompression!=BI_RGB
				&& pbmih->biCompression!=BI_RLE4
				&& pbmih->biCompression!=BI_RLE8
				&& pbmih->biCompression!=BI_BITFIELDS)
			|| (pbmih->biCompression==BI_RLE4 && pbmih->biBitCount!=4)
			|| (pbmih->biCompression==BI_RLE8 && pbmih->biBitCount!=8)
			|| (pbmih->biCompression==BI_BITFIELDS
				&& (pbmih->biBitCount!=16 && pbmih->biBitCount!=32))
			|| (pbmih->biSizeImage==0
				&& (pbmih->biCompression!=BI_RGB
					&& pbmih->biCompression!=BI_BITFIELDS)))
		return 1;
	PixelBytes=DIB_ROW_BYTES(pbmih->biWidth,pbmih->biBitCount)*
														abs(pbmih->biHeight);
	if (pbmih->biSizeImage!=0
			&& (pbmih->biCompression==BI_RGB
				|| pbmih->biCompression==BI_BITFIELDS)
			&& pbmih->biSizeImage!=PixelBytes) {
		if (pbmih->biWidth*2==pbmih->biHeight
				&& pbmih->biSizeImage==PixelBytes/2+
						DIB_ROW_BYTES(pbmih->biWidth,1)*(pbmih->biHeight/2)) {
			fIcon=TRUE;
		} else {
			return 1;
		}
	}
	Size=sizeof(BITMAPINFOHEADER);
	if (pbmih->biBitCount<=8) {
		if (pbmih->biClrUsed==0
				|| pbmih->biClrUsed==1UL<<pbmih->biBitCount) {
			Size+=(DataSizeType)(1<<pbmih->biBitCount)*sizeof(RGBQUAD);
		} else {
			return sizeof(BITMAPINFOHEADER);
		}
	}
	if (!fIcon && pbmih->biSizeImage==0
			&& (pbmih->biCompression==BI_RGB
				|| pbmih->biCompression==BI_BITFIELDS)
			&& pbmih->biWidth*2==pbmih->biHeight) {
		size_t IconSize=Size+PixelBytes/2+
						DIB_ROW_BYTES(pbmih->biWidth,1)*(pbmih->biHeight/2);
		const BYTE *p=pData+IconSize;

		if ((DataSize>=IconSize && DataSize<Size+PixelBytes)
				|| (DataSize>=IconSize+4
					&& p[0]==sizeof(BITMAPINFOHEADER)
					&& p[1]==0 && p[2]==0 && p[3]==0))
			fIcon=TRUE;
	}
	if (!fIcon) {
		if (pbmih->biCompression==BI_RLE4 || pbmih->biCompression==BI_RLE8) {
			Size+=pbmih->biSizeImage;
		} else {
			if (pbmih->biCompression==BI_BITFIELDS)
				Size+=3*sizeof(DWORD);
			Size+=PixelBytes;
		}
	} else {
		Size+=PixelBytes/2+DIB_ROW_BYTES(pbmih->biWidth,1)*(pbmih->biHeight/2);
	}
	if (Size>DataSize)
		return sizeof(BITMAPINFOHEADER);
	if (!(*pfnDataCallback)(fIcon?FORMAT_ICON:FORMAT_DIB,pData,Size,pParam))
		return 0;
	return Size;
}


static DataSizeType ExtractJPEGData(const BYTE *pData,DataSizeType DataSize,
							ExtractDataCallback pfnDataCallback,void *pParam)
{
	const BYTE *p;
	DataSizeType Size,Remain;
	BOOL fHasSOF,fHasSOS;
	int Marker;

	p=pData+2;
	if (p[0]!=0xFF || p[1]!=JPEG_MARKER_APP0)
		return 1;
	p+=2;
	Size=(p[0]<<8)|p[1];
	if (Size<5 || 2+2+Size>=DataSize)
		return 1;
	p+=2;
	if (p[0]!='J' || p[1]!='F' || p[2]!='I' || p[3]!='F')
		return 1;
	fHasSOF=FALSE;
	fHasSOS=FALSE;
	p+=Size-2;
	Remain=DataSize-(2+2+Size);
	while (TRUE) {
		if (Remain<2 || *p!=0xFF)
			return 1;
		p++;
		Marker=*p++;
		if (Marker==JPEG_MARKER_EOI) {
			if (!fHasSOF || !fHasSOS)
				return 1;
			break;
		}
		if (Marker==JPEG_MARKER_SOF0 || Marker==JPEG_MARKER_SOF2) {
			fHasSOF=TRUE;
		} else if (Marker==JPEG_MARKER_SOF1 || Marker==JPEG_MARKER_SOF3
				|| (Marker>=JPEG_MARKER_SOF5 && Marker<=JPEG_MARKER_SOF7)
				|| (Marker>=JPEG_MARKER_SOF9 && Marker<=JPEG_MARKER_SOF11)) {
			/* Unsupported */
			return 1;
		}
		Remain-=2;
		Size=(p[0]<<8)|p[1];
		if (Size>=Remain)
			return 1;
		p+=Size;
		Remain-=Size;
		if (Marker==JPEG_MARKER_SOS) {
			while (TRUE) {
				if (Remain<2)
					return 1;
				if (*p==0xFF) {
					if (p[1]==0xFF
							|| (p[1]>=JPEG_MARKER_RST0
												&& p[1]<=JPEG_MARKER_RST7)) {
						p+=2;
						Remain-=2;
					} else if (p[1]<=0xBF) {
						p++;
						Remain--;
					} else
						break;
				} else {
					p++;
					Remain--;
				}
			}
			fHasSOS=TRUE;
		}
	}
	Size=p-pData;
	if (!(*pfnDataCallback)(FORMAT_JPEG,pData,Size,pParam))
		return 0;
	return Size;
}


static DataSizeType ExtractPNGData(const BYTE *pData,DataSizeType DataSize,
							ExtractDataCallback pfnDataCallback,void *pParam)
{
	const BYTE *p;
	DataSizeType RemainSize,Size;
	unsigned int Width,Height;
	int BitsPerPlane,ColorType,Compression,Filter,Interlace;
	BOOL fHasImageData;

	p=pData+8;
	Size=MSBFirst32(p);
	p+=4;
	if (Size!=13 || p[0]!='I' || p[1]!='H' || p[2]!='D' || p[3]!='R')
		return 8;
	p+=4;
	Width=MSBFirst32(p);
	p+=4;
	Height=MSBFirst32(p);
	p+=4;
	BitsPerPlane=p[0];
	ColorType=p[1];
	Compression=p[2];
	Filter=p[3];
	Interlace=p[4];
	if (Width==0 || Height==0
			|| (BitsPerPlane!=1 && BitsPerPlane!=2 && BitsPerPlane!=4
				&& BitsPerPlane!=8 && BitsPerPlane!=16)
			|| (ColorType!=0 && ColorType!=2 && ColorType!=3 && ColorType!=4
				&& ColorType!=6)
			|| Compression!=0 || Filter!=0 || Interlace>1)
		return 8;
	p+=5+4;
	RemainSize=DataSize-(8+4+4+13+4);
	fHasImageData=FALSE;
	while (TRUE) {
		if (RemainSize<4+4+4)
			return 8;
		Size=MSBFirst32(p);
		if (4+4+Size+4>RemainSize)
			return 8;
		p+=4;
		if (p[0]=='I' && p[1]=='E' && p[2]=='N' && p[3]=='D') {
			if (!fHasImageData || Size!=0)
				return 8;
			break;
		}
		if (p[0]=='I' && p[1]=='D' && p[2]=='A' && p[3]=='T')
			fHasImageData=TRUE;
		p+=4+Size+4;
		RemainSize-=4+4+Size+4;
	}
	p+=4+4;
	Size=p-pData;
	if (!(*pfnDataCallback)(FORMAT_PNG,pData,Size,pParam))
		return 0;
	return Size;
}


static DataSizeType ExtractMAGData(const BYTE *pData,DataSizeType DataSize,
							ExtractDataCallback pfnDataCallback,void *pParam)
{
	const BYTE *p;
	DataSizeType HeaderOffset,Size;
	const MagHeader *pmh;

	p=pData+8;
	while (p<pData+DataSize-sizeof(MagHeader) && *p++!=0x1A);
	if (p==pData+DataSize+sizeof(MagHeader))
		return 8;
	HeaderOffset=p-pData;
	pmh=(const MagHeader*)p;
	if (pmh->bDummy!=0 || pmh->wXStart>pmh->wXEnd || pmh->wYStart>pmh->wYEnd
			|| pmh->dwFlagAOffset!=sizeof(MagHeader)+
								((pmh->bScreenMode&MAG_256COLORS)!=0?256:16)*3
			|| pmh->dwFlagBOffset<=pmh->dwFlagAOffset
			|| pmh->dwFlagBOffset>DataSize-HeaderOffset
			|| pmh->dwFlagBSize==0
			|| pmh->dwPixelOffset!=pmh->dwFlagBOffset+pmh->dwFlagBSize
			|| pmh->dwPixelOffset+HeaderOffset>DataSize
			|| pmh->dwPixelSize==0
			|| pmh->dwPixelSize+pmh->dwPixelOffset+HeaderOffset>DataSize)
		return 8;
	Size=HeaderOffset+pmh->dwPixelOffset+pmh->dwPixelSize;
	if (!(*pfnDataCallback)(FORMAT_MAG,pData,Size,pParam))
		return 0;
	return Size;
}


static DataSizeType ExtractIconData(const BYTE *pData,DataSizeType DataSize,
							ExtractDataCallback pfnDataCallback,void *pParam)
{
	const BYTE *p;
	const ICONDIR *pid;
	BOOL fIcon;
	int Count,i;
	const ICONDIRENTRY *pide;
	const BITMAPINFOHEADER *pbmih;
	DataSizeType Size;

	p=pData;
	pid=(const ICONDIR*)p;
	fIcon=pid->idType==1;
	Count=pid->idCount;
	pide=(const ICONDIRENTRY*)(p+sizeof(ICONDIR));
	for (i=0;i<Count;i++) {
		if (/*pide[i].bWidth==0 || pide[i].bHeight==0
				||*/ (fIcon && pide[i].wBitCount!=0 && pide[i].wBitCount!=1
					&& pide[i].wBitCount!=4 && pide[i].wBitCount!=8
					&& pide[i].wBitCount!=24 && pide[i].wBitCount!=32)
				|| pide[i].dwBytesInRes<sizeof(BITMAPINFOHEADER)+4+4
				|| pide[i].dwImageOffset<
									sizeof(ICONDIR)+sizeof(ICONDIRENTRY)*Count
				|| pide[i].dwImageOffset>=DataSize
				|| pide[i].dwBytesInRes>DataSize-pide[i].dwImageOffset)
			return 1;
	}
	for (i=0;i<Count;i++) {
		/* TODO: PNG format support */
		pbmih=(const BITMAPINFOHEADER*)(p+pide[i].dwImageOffset);
		if (pbmih->biSize!=sizeof(BITMAPINFOHEADER)
				|| pbmih->biWidth<=0 || pbmih->biHeight<=0
				|| pbmih->biPlanes!=1
				|| (pbmih->biBitCount!=1 && pbmih->biBitCount!=4
					&& pbmih->biBitCount!=8 && pbmih->biBitCount!=24
					&& pbmih->biBitCount!=32)
				|| pbmih->biCompression!=BI_RGB)
			return 1;
	}
	Size=pide[Count-1].dwImageOffset+pide[Count-1].dwBytesInRes;
	if (!(*pfnDataCallback)(fIcon?FORMAT_ICON:FORMAT_CURSOR,pData,Size,pParam))
		return 0;
	return Size;
}


#ifdef __BORLANDC__
#pragma argsused
#endif
static DataSizeType ExtractANIData(const BYTE *pData,DataSizeType DataSize,
							ExtractDataCallback pfnDataCallback,void *pParam)
{
	const BYTE *p;
	DataSizeType RIFFSize,RemainSize,Size,ListSize;
	BOOL fHasHeader,fHasIcon;
	const ANIHEADER *panih;

	p=pData+4;
	RIFFSize=(LSBFirst32(p)+1)/2*2;
	p+=4+4;
	RemainSize=RIFFSize-4;
	if (p[0]=='L' && p[1]=='I' && p[2]=='S' && p[3]=='T') {
		p+=4;
		ListSize=(LSBFirst32(p)+1)/2*2;
		RemainSize-=4+4;
		if (ListSize<4 || ListSize>RemainSize)
			return 1;
		p+=4;
		if (p[0]!='I' || p[1]!='N' || p[2]!='F' || p[3]!='O')
			return 1;
		p+=4;
		ListSize-=4;
		RemainSize-=4;
		while (ListSize>0) {
			if (ListSize<4+4)
				return 1;
#if 0
			if ((p[0]!='I' || p[1]!='N' || p[2]!='A' || p[3]!='M')
					&& (p[0]!='I' || p[1]!='A' || p[2]!='R' || p[3]!='T'))
				return 1;
#endif
			p+=4;
			Size=(LSBFirst32(p)+1)/2*2;
			ListSize-=4+4;
			if (Size>ListSize)
				return 1;
			ListSize-=Size;
			p+=4+Size;
			RemainSize-=4+4+Size;
		}
	}
	fHasHeader=FALSE;
	fHasIcon=FALSE;
	while (RemainSize>0) {
		if (RemainSize<4+4)
			return 1;
		Size=(LSBFirst32(p+4)+1)/2*2;
		RemainSize-=4+4;
		if (RemainSize<Size)
			return 1;
		if (p[0]=='a' && p[1]=='n' && p[2]=='i' && p[3]=='h') {
			if (Size!=sizeof(ANIHEADER))
				return 1;
			p+=4+4;
			panih=(const ANIHEADER*)p;
			if (panih->cbSizeOf!=sizeof(ANIHEADER)
					|| panih->cFrames==0 || panih->cSteps==0)
				return 1;
			p+=sizeof(ANIHEADER);
			RemainSize-=Size;
			fHasHeader=TRUE;
		} else if (p[0]=='L' && p[1]=='I' && p[2]=='S' && p[3]=='T') {
			const ICONDIR *pid;

			RemainSize-=Size;
			ListSize=Size;
			p+=4+4;
			if (p[0]!='f' || p[1]!='r' || p[2]!='a' || p[3]!='m')
				return 1;
			p+=4;
			ListSize-=4;
			while (ListSize>0) {
				if (ListSize<4+4)
					return 1;
				if (p[0]!='i' || p[1]!='c' || p[2]!='o' || p[3]!='n')
					return 1;
				p+=4;
				Size=(LSBFirst32(p)+1)/2*2;
				ListSize-=4+4;
				if (ListSize<Size)
					return 1;
				p+=4;
				pid=(const ICONDIR*)p;
				if (pid->idReserved!=0 || (pid->idType!=1 && pid->idType!=2)
															|| pid->idCount==0)
					return 1;
				p+=Size;
				ListSize-=Size;
			}
			fHasIcon=TRUE;
		} else if ((p[0]=='r' && p[1]=='a' && p[2]=='t' && p[3]=='e')
				|| (p[0]=='s' && p[1]=='e' && p[2]=='q' && p[3]==' ')) {
			p+=4+4+Size;
			RemainSize-=Size;
		} else
			return 1;
	}
	if (!fHasHeader || !fHasIcon)
		return 1;
	Size=RIFFSize+(4+4);
	if (!(*pfnDataCallback)(FORMAT_ANI,pData,Size,pParam))
		return 0;
	return Size;
}


static DataSizeType ExtractWAVEData(const BYTE *pData,DataSizeType DataSize,
							ExtractDataCallback pfnDataCallback,void *pParam)
{
	const BYTE *p;
	DWORD dwRiffSize,dwWaveSize,dwHeaderSize,dwDataSize;
	const WAVEFORMATEX *pwfx;

	p=pData+4;
	dwRiffSize=LSBFirst32(p);
	if (dwRiffSize>DataSize-8 || dwRiffSize<MIN_WAVE_FILE_SIZE-8)
		return 4;
	p+=4;
#if 0
	if (p[0]!='W' || p[1]!='A' || p[2]!='V' || p[3]!='E')
		return 4;
#endif
	p+=4;
	if (p[0]!='f' || p[1]!='m' || p[2]!='t' || p[3]!=' ')
		return 12;
	p+=4;
	dwHeaderSize=LSBFirst32(p);
	if (dwHeaderSize>DataSize-8-4-8-8-2|| dwHeaderSize<sizeof(WAVEFORMATEX)-2)
		return 16;
	p+=4;
	pwfx=(const WAVEFORMATEX*)p;
	if ((pwfx->wFormatTag==WAVE_FORMAT_PCM
				&& (dwHeaderSize!=sizeof(WAVEFORMATEX)-2
									/* Ȃ PCM ł WAVEFORMATEX Ȃ̂
												( cbSize C`L) */
										&& dwHeaderSize!=sizeof(WAVEFORMATEX)))
			|| (pwfx->wFormatTag!=WAVE_FORMAT_PCM
							&& dwHeaderSize!=sizeof(WAVEFORMATEX)+pwfx->cbSize)
			|| pwfx->nChannels==0
			|| pwfx->nSamplesPerSec==0
			|| pwfx->wBitsPerSample==0)
		return 16;
	p+=dwHeaderSize;
	dwWaveSize=8+4+8+dwHeaderSize;
	if (p[0]=='f') {
		if (p[0]!='f' || p[1]!='a' || p[2]!='c' || p[3]!='t')
			return 16;
		p+=4;
		if (*(DWORD*)p!=4 || *(DWORD*)(p+4)==0)
			return dwWaveSize+4;
		p+=8;
		dwWaveSize+=12;
	}
	if (p[0]!='d' || p[1]!='a' || p[2]!='t' || p[3]!='a') {
		if (pwfx->wFormatTag==WAVE_FORMAT_PCM)
			return 16;
		return dwWaveSize-8;
	}
	p+=4;
	dwDataSize=LSBFirst32(p);
	dwWaveSize+=8;
	if (dwDataSize==0 || dwWaveSize+(dwDataSize+1)/2*2>DataSize
								|| dwWaveSize+(dwDataSize+1)/2*2!=dwRiffSize+8)
		return dwWaveSize-8;
	dwWaveSize+=(dwDataSize+1)/2*2;
	if (!(*pfnDataCallback)(FORMAT_WAVE,pData,dwWaveSize,pParam))
		return 0;
	return dwWaveSize;
}


static DataSizeType ExtractMIDIData(const BYTE *pData,DataSizeType DataSize,
							ExtractDataCallback pfnDataCallback,void *pParam)
{
	const BYTE *p;
	int NumTracks;
	DataSizeType RemainSize,Size;
	int i;

	if (DataSize<=4+4+6)
		return 4;
	p=pData;
	p+=4;
	if (MSBFirst32(p)!=6			/* Data length */
			|| p[4]!=0 || p[5]>2)	/* Format */
		return 4;
	NumTracks=(p[6]<<8)|p[7];
	if (NumTracks==0)
		return 4;
	p+=4+6;
	RemainSize=DataSize-4+4+6;
	for (i=0;i<NumTracks;i++) {
		if (RemainSize<8
				|| p[0]!='M' || p[1]!='T' || p[2]!='r' || p[3]!='k')
			return 4;
		p+=4;
		Size=MSBFirst32(p);
		RemainSize-=8;
		if (Size>RemainSize)
			return 4;
		p+=4+Size;
		RemainSize-=Size;
	}
	Size=p-pData;
	if (!(*pfnDataCallback)(FORMAT_MIDI,pData,Size,pParam))
		return 0;
	return Size;
}


static DataSizeType ExtractMP3Data(const BYTE *pData,DataSizeType DataSize,
							ExtractDataCallback pfnDataCallback,void *pParam)
{
	BOOL fID3v2;
	const BYTE *p,*q;
	DataSizeType RemainSize,Size;
	MP3FrameInfo FrameInfo;
	DWORD NumFrames,FrameBytes;

	RemainSize=DataSize;
	p=pData;
	fID3v2=*p=='I';
	if (fID3v2) {
		/* ID3 v2 */
		if (RemainSize<=3+3+4)
			return 1;
		p+=3+3;
		if ((p[0]&0x80)!=0 || (p[1]&0x80)!=0 || (p[2]&0x80)!=0 || (p[3]&0x80)!=0)
			return 1;
		Size=(p[0]<<(7*3))|(p[1]<<(7*2))|(p[2]<<7)|p[3];
		RemainSize-=3+3+4;
		if (Size+4>=RemainSize)
			return 1;
		p+=4+Size;
		if (*p!=0xFF || (*(p+1)&0xE0)!=0xE0)
			return 1;
		RemainSize-=Size;
	}
	if (!MP3GetFrameInfo(p,&FrameInfo) || RemainSize<FrameInfo.FrameSize)
		return 1;
	Size=RemainSize-4;
	NumFrames=FrameBytes=0;
	q=p+4;
	do {
		if (Size<4+4+4+4)
			break;
		if ((q[0]=='X' && q[1]=='i' && q[2]=='n' && q[3]=='g')
				|| (q[0]=='I' && q[1]=='n' && q[2]=='f' && q[3]=='o')) {
			q+=4;
			if ((q[3]&0x03)==0x03) {
				q+=4;
				NumFrames=MSBFirst32(q);
				q+=4;
				FrameBytes=MSBFirst32(q);
				//q+=4;
			}
			break;
		} else if (q[0]=='V' && q[1]=='B' && q[2]=='R' && q[3]=='I') {
			q+=4+2+2+2;
			FrameBytes=MSBFirst32(q);
			q+=4;
			NumFrames=MSBFirst32(q);
			//q+=4;
			break;
		}
		q++;
		Size--;
	} while (q<p+FrameInfo.FrameSize);
	if (NumFrames==0 || FrameBytes<=FrameInfo.FrameSize) {
		WORD PrevFrameSize;

		/*
			Xing(Info)  VBRI wb_ꍇ́A댟oh~̂
			ID3 v2 ꍇ̂ݑΏۂɂ
		*/
		if (!fID3v2 || FrameBytes!=0 || q-p!=FrameInfo.FrameSize
				|| q[0]!=0xFF || (q[1]&0xE0)!=0xE0)
			return 1;
		FrameBytes=PrevFrameSize=FrameInfo.FrameSize;
		Size=RemainSize-FrameInfo.FrameSize;
		while (Size>=MIN_MP3_FRAME_SIZE) {
			MP3FrameInfo CurFrame;

			if (q[0]!=0xFF || (q[1]&0xE0)!=0xE0
					|| !MP3GetFrameInfo(q,&CurFrame)
					|| Size<CurFrame.FrameSize
					|| CurFrame.Version!=FrameInfo.Version
					|| CurFrame.Layer!=FrameInfo.Layer
					|| CurFrame.Frequency!=FrameInfo.Frequency
					|| CurFrame.ChannelMode!=FrameInfo.ChannelMode) {
				if (FrameBytes>PrevFrameSize) {
					const BYTE *r=q-PrevFrameSize;
					WORD i;

					Size+=PrevFrameSize-4;
					for (i=4;i<PrevFrameSize && Size>=128;i++) {
						if (r[i]=='T' && r[i+1]=='A' && r[i+2]=='G') {
							FrameBytes-=PrevFrameSize-i;
							break;
						}
						Size--;
					}
				}
				break;
			}
			FrameBytes+=CurFrame.FrameSize;
			Size-=CurFrame.FrameSize;
			q+=CurFrame.FrameSize;
			PrevFrameSize=CurFrame.FrameSize;
		}
	}
	if (RemainSize<FrameBytes)
		return 1;
	RemainSize-=FrameBytes;
	p+=FrameBytes;
	if (RemainSize>=128 && p[0]=='T' && p[1]=='A' && p[2]=='G') {
		/* ID3 v1 */
		//p+=128;
		RemainSize-=128;
	}
	Size=DataSize-RemainSize;
	if (!(*pfnDataCallback)(FORMAT_MP3,pData,Size,pParam))
		return 0;
	return Size;
}


#define NOCOMPMAN
#include <vfw.h>

static DataSizeType ExtractAVIData(const BYTE *pData,DataSizeType DataSize,
							ExtractDataCallback pfnDataCallback,void *pParam)
{
	const BYTE *p;
	DWORD RiffSize,ListSize,AVIHeaderSize;
	DataSizeType AVISize;
	const MainAVIHeader *pmavih;

	if (DataSize<=32+56)
		return 4;
	p=pData+4;
	RiffSize=LSBFirst32(p);
	if (RiffSize>DataSize-8)
		return 4;
	p+=4;
#if 0
	if (p[0]!='A' || p[1]!='V' || p[2]!='I' || p[3]!=' ')
		return 4;
#endif
	p+=4;
	if (p[0]!='L' || p[1]!='I' || p[2]!='S' || p[3]!='T')
		return 12;
	p+=4;
	ListSize=LSBFirst32(p);
	if (ListSize>=DataSize-12-12)
		return 16;
	p+=4;
	if (p[0]!='h' || p[1]!='d' || p[2]!='r' || p[3]!='l')
		return 16;
	p+=4;
	if (p[0]!='a' || p[1]!='v' || p[2]!='i' || p[3]!='h')
		return 24;
	p+=4;
	AVIHeaderSize=LSBFirst32(p);
	if (AVIHeaderSize!=56)
		return 28;
	p+=4;
	pmavih=(const MainAVIHeader*)p;
	if (pmavih->dwStreams==0 || pmavih->dwWidth==0 || pmavih->dwHeight==0)
		return 28;
	AVISize=RiffSize+8;
	if (!(*pfnDataCallback)(FORMAT_AVI,pData,AVISize,pParam))
		return 0;
	return AVISize;
}


/*
	f[^𒊏o
*/
BOOL ExtractData(const void *pData,DataSizeType DataSize,
					ExtractDataCallback pfnDataCallback,
					ExtractProgressCallback pfnProgressCallback,void *pParam)
{
	const BYTE *pCurData;
	DataSizeType RemainSize,Size;
	int PrevPos,CurPos;

	pCurData=(const BYTE*)pData;
	RemainSize=DataSize;
	PrevPos=-1;
	while (RemainSize>=MIN_FILE_SIZE) {
#ifdef _DEBUG
		if (pCurData>=(const BYTE*)pData+DataSize) {
			MESSAGE(TEXT("ExtractData() : Invalid data pos %p / %p + ") TEXT(PRINTF_SIZE),
					pCurData,pData,RemainSize);
			return FALSE;
		}
#endif
		CurPos=(int)((pCurData-(const BYTE*)pData)/PROGRESS_BYTES);
		if (CurPos>PrevPos) {
			if (!(*pfnProgressCallback)(CurPos,(int)(DataSize/PROGRESS_BYTES),
																	pParam))
				return FALSE;
			PrevPos=CurPos;
		}
		if (RemainSize>=MIN_DIB_FILE_SIZE
				&& pCurData[0]=='B' && pCurData[1]=='M') {
			Size=ExtractDIBFileData(pCurData,RemainSize,pfnDataCallback,
																	pParam);
		} else if (RemainSize>=MIN_DIB_SIZE
				&& pCurData[0]==sizeof(BITMAPINFOHEADER)
				&& pCurData[1]==0 && pCurData[2]==0 && pCurData[3]==0) {
			Size=ExtractDIBData(pCurData,RemainSize,pfnDataCallback,pParam);
		} else if (pCurData[0]==0xFF && pCurData[1]==JPEG_MARKER_SOI) {
			Size=ExtractJPEGData(pCurData,RemainSize,pfnDataCallback,pParam);
		} else if (pCurData[0]==0x89 && pCurData[1]=='P'
				&& pCurData[2]=='N' && pCurData[3]=='G'
				&& pCurData[4]=='\r' && pCurData[5]=='\n'
				&& pCurData[6]==0x1A && pCurData[7]=='\n') {
			Size=ExtractPNGData(pCurData,RemainSize,pfnDataCallback,pParam);
		} else if (pCurData[0]=='M' && pCurData[1]=='A'
				&& pCurData[2]=='K' && pCurData[3]=='I'
				&& pCurData[4]=='0' && pCurData[5]=='2'
				&& pCurData[6]==' ' && pCurData[7]==' ') {
			Size=ExtractMAGData(pCurData,RemainSize,pfnDataCallback,pParam);
		} else if (pCurData[0]=='R' && pCurData[1]=='I'
				&& pCurData[2]=='F' && pCurData[3]=='F'
				&& LSBFirst32(pCurData+4)<=RemainSize-8) {
			if (pCurData[8]=='A' && pCurData[9]=='C'
					&& pCurData[10]=='O' && pCurData[11]=='N') {
				Size=ExtractANIData(pCurData,RemainSize,pfnDataCallback,
																	pParam);
			} else if (RemainSize>=MIN_WAVE_FILE_SIZE
					&& pCurData[8]=='W' && pCurData[9]=='A'
					&& pCurData[10]=='V' && pCurData[11]=='E') {
				Size=ExtractWAVEData(pCurData,RemainSize,pfnDataCallback,
																	pParam);
			} else if (pCurData[8]=='A' && pCurData[9]=='V'
					&& pCurData[10]=='I' && pCurData[11]==' ') {
				Size=ExtractAVIData(pCurData,RemainSize,pfnDataCallback,
																	pParam);
			} else {
				pCurData+=4;
				RemainSize-=4;
				continue;
			}
		} else if (pCurData[0]=='M' && pCurData[1]=='T'
				&& pCurData[2]=='h' && pCurData[3]=='d') {
			Size=ExtractMIDIData(pCurData,RemainSize,pfnDataCallback,pParam);
		} else if (RemainSize>=MIN_MP3_FRAME_SIZE*10
				&& ((pCurData[0]=='I' && pCurData[1]=='D' && pCurData[2]=='3')
				|| (pCurData[0]==0xFF && (pCurData[1]&0xE0)==0xE0))) {
			Size=ExtractMP3Data(pCurData,RemainSize,pfnDataCallback,pParam);
		} else if (RemainSize>sizeof(ICONDIR)
				&& pCurData[0]==0 && pCurData[1]==0
				&& (pCurData[2]==1 || pCurData[2]==2) && pCurData[3]==0
				&& pCurData[4]>0 && pCurData[5]==0
				&& RemainSize>
							sizeof(ICONDIR)+sizeof(ICONDIRENTRY)*pCurData[4]) {
			Size=ExtractIconData(pCurData,RemainSize,pfnDataCallback,pParam);
		} else {
			pCurData++;
			RemainSize--;
			continue;
		}
		if (Size==0)	/* Abort */
			return FALSE;
#ifdef _DEBUG
		if (Size>RemainSize) {
			MESSAGE(TEXT("ExtractData() : Data size too large"));
			return FALSE;
		}
#endif
		pCurData+=Size;
		RemainSize-=Size;
	}
	return TRUE;
}
