/******************************************************************************
*                                                                             *
*    Formats.c                              Copyright(c) 2005-2010 itow,y.    *
*                                                                             *
******************************************************************************/

/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <windows.h>
#include <tchar.h>
#include "DataCutter.h"
#include "Formats.h"
#include "JPEG.h"
#include "PNGDecode.h"




ErrorType DataSaveToFile(LPCTSTR pszFileName,const void *pData,size_t DataSize)
{
	HANDLE hFile;
	DWORD Wrote;

	hFile=CreateFile(pszFileName,GENERIC_WRITE,0,NULL,CREATE_ALWAYS,
												FILE_ATTRIBUTE_NORMAL,NULL);
	if (hFile==INVALID_HANDLE_VALUE)
		return ERR_FILEOPEN;
#ifndef WIN64
	if (!WriteFile(hFile,pData,DataSize,&Wrote,NULL) || Wrote!=DataSize) {
		CloseHandle(hFile);
		return ERR_FILEWRITE;
	}
#else
	while (DataSize>0) {
		DWORD Size=(DWORD)min(DataSize,(size_t)0x7FFFFFFF);
		if (!WriteFile(hFile,pData,Size,&Wrote,NULL) || Wrote!=Size) {
			CloseHandle(hFile);
			return ERR_FILEWRITE;
		}
		DataSize-=Size;
	}
#endif
	if (!CloseHandle(hFile))
		return ERR_FILECLOSE;
	return ERR_SUCCESS;
}


ErrorType RiffSaveToFile(LPCTSTR pszFileName,const void *pData,size_t DataSize)
{
	DWORD RiffSize,Wrote;
	HANDLE hFile;

	if (DataSize<8)
		return ERR_INTERNAL;
	RiffSize=LSBFirst32((const BYTE*)pData+4);
	if (RiffSize>0xFFFFFFFFUL-8)
		return ERR_OTHER;
	RiffSize+=8;
	if (RiffSize>DataSize)
		return ERR_INTERNAL;
	hFile=CreateFile(pszFileName,GENERIC_WRITE,FILE_SHARE_READ,NULL,
									CREATE_ALWAYS,FILE_ATTRIBUTE_NORMAL,NULL);
	if (hFile==INVALID_HANDLE_VALUE)
		return ERR_FILEOPEN;
	if (!WriteFile(hFile,pData,RiffSize,&Wrote,NULL) || Wrote!=RiffSize) {
		CloseHandle(hFile);
		return ERR_FILEWRITE;
	}
	if (!CloseHandle(hFile))
		return ERR_FILECLOSE;
	return ERR_SUCCESS;
}




/******************************************************************************

	DIB֐

******************************************************************************/


/*
	DIBt@Cɕۑ
*/
ErrorType DIBSaveToFile(LPCTSTR pszFileName,const void *pData,size_t DataSize)
{
	HANDLE hFile;
	const BITMAPFILEHEADER *pbmfhSrc;
	const BITMAPINFO *pbmiSrc;
	const void *pBits;
	UINT InfoBytes,BitsBytes;
	BITMAPFILEHEADER bmfh;
	BITMAPINFOHEADER bmih;
	DWORD dwWrite;

	hFile=CreateFile(pszFileName,GENERIC_WRITE,0,NULL,CREATE_ALWAYS,
												FILE_ATTRIBUTE_NORMAL,NULL);
	if (hFile==INVALID_HANDLE_VALUE)
		return ERR_FILEOPEN;
	if (((BYTE*)pData)[0]=='B' && ((BYTE*)pData)[1]=='M') {
		pbmfhSrc=(const BITMAPFILEHEADER*)pData;
		pbmiSrc=(const BITMAPINFO*)((BYTE*)pData+sizeof(BITMAPFILEHEADER));
	} else {
		pbmfhSrc=NULL;
		pbmiSrc=(const BITMAPINFO*)pData;
	}
	/* TCY߂ */
	InfoBytes=DIBCalcInfoBytes(&pbmiSrc->bmiHeader);
	if (pbmiSrc->bmiHeader.biCompression==BI_RGB
			|| pbmiSrc->bmiHeader.biCompression==BI_BITFIELDS)
		BitsBytes=DIBCalcBitsBytes(&pbmiSrc->bmiHeader);
	else
		BitsBytes=pbmiSrc->bmiHeader.biSizeImage;
	/* BITMAPFILEHEADER ̏ */
	bmfh.bfType=0x4D42;
	bmfh.bfSize=sizeof(BITMAPFILEHEADER)+InfoBytes+BitsBytes;
	bmfh.bfReserved1=0;
	bmfh.bfReserved2=0;
	bmfh.bfOffBits=sizeof(BITMAPFILEHEADER)+InfoBytes;
	if (!WriteFile(hFile,&bmfh,sizeof(BITMAPFILEHEADER),&dwWrite,NULL)
										|| dwWrite!=sizeof(BITMAPFILEHEADER)) {
		CloseHandle(hFile);
		return ERR_FILEWRITE;
	}
	/* BITMAPINFOHEADER ̏ */
	bmih.biSize=sizeof(BITMAPINFOHEADER);
	bmih.biWidth=pbmiSrc->bmiHeader.biWidth;
	bmih.biHeight=pbmiSrc->bmiHeader.biHeight;
	bmih.biPlanes=1;
	bmih.biBitCount=pbmiSrc->bmiHeader.biBitCount;
	bmih.biCompression=pbmiSrc->bmiHeader.biCompression;
	bmih.biSizeImage=pbmiSrc->bmiHeader.biCompression==BI_RGB
						|| pbmiSrc->bmiHeader.biCompression==BI_BITFIELDS?0:
												pbmiSrc->bmiHeader.biSizeImage;
	bmih.biXPelsPerMeter=pbmiSrc->bmiHeader.biXPelsPerMeter;
	bmih.biYPelsPerMeter=pbmiSrc->bmiHeader.biYPelsPerMeter;
	bmih.biClrUsed=pbmiSrc->bmiHeader.biClrUsed;
	bmih.biClrImportant=pbmiSrc->bmiHeader.biClrImportant;
	if (!WriteFile(hFile,&bmih,sizeof(BITMAPINFOHEADER),&dwWrite,NULL)
										|| dwWrite!=sizeof(BITMAPINFOHEADER)) {
		CloseHandle(hFile);
		return ERR_FILEWRITE;
	}
	/* pbg̏ */
	if (pbmiSrc->bmiHeader.biBitCount<=8) {
		int nColors,nPalColors,i;
		RGBQUAD rgb[256];

		nColors=nPalColors=1<<pbmiSrc->bmiHeader.biBitCount;
		if (pbmfhSrc!=NULL && pbmiSrc->bmiHeader.biClrUsed!=0
				&& pbmiSrc->bmiHeader.biClrUsed!=(unsigned)nColors
				&& pbmiSrc->bmiHeader.biClrUsed*sizeof(RGBQUAD)==
								pbmfhSrc->bfOffBits-sizeof(BITMAPFILEHEADER)-
													sizeof(BITMAPINFOHEADER))
			nPalColors=pbmiSrc->bmiHeader.biClrUsed;
		for (i=0;i<nPalColors;i++)
			rgb[i]=pbmiSrc->bmiColors[i];
		if (nPalColors<nColors)
			MemZero(&rgb[i],(nColors-nPalColors)*sizeof(RGBQUAD));
		if (!WriteFile(hFile,rgb,nColors*sizeof(RGBQUAD),&dwWrite,NULL)
										|| dwWrite!=nColors*sizeof(RGBQUAD)) {
			CloseHandle(hFile);
			return ERR_FILEWRITE;
		}
	}
	/* f[^̏ */
	pBits=(const BYTE*)pbmiSrc+DIBGetInfoBytes(pData);
	if (!WriteFile(hFile,pBits,BitsBytes,&dwWrite,NULL)
													|| dwWrite!=BitsBytes) {
		CloseHandle(hFile);
		return ERR_FILEWRITE;
	}
	if (!CloseHandle(hFile))
		return ERR_FILECLOSE;
	return ERR_SUCCESS;
}


/*
	DIBNbv{[hɃRs[
*/
ErrorType DIBCopyToClipboard(HWND hwnd,const void *pData,size_t DataSize)
{
	const BITMAPINFO *pbmiSrc;
	const void *pBits;
	UINT SrcInfoBytes,InfoBytes,BitsBytes;
	HGLOBAL hglobal;
	BYTE *pbData;
	BITMAPINFOHEADER *pbmih;

	if (((BYTE*)pData)[0]=='B' && ((BYTE*)pData)[1]=='M')
		pbmiSrc=(BITMAPINFO*)((BYTE*)pData+sizeof(BITMAPFILEHEADER));
	else
		pbmiSrc=(BITMAPINFO*)pData;
	InfoBytes=DIBCalcInfoBytes(&pbmiSrc->bmiHeader);
	BitsBytes=DIBCalcBitsBytes(&pbmiSrc->bmiHeader);
	hglobal=GlobalAlloc(GMEM_MOVEABLE | GMEM_SHARE,InfoBytes+BitsBytes);
	if (hglobal==NULL)
		return ERR_MEMORYALLOC;
	pbData=(BYTE*)GlobalLock(hglobal);
	if (pbData==NULL) {
		GlobalFree(hglobal);
		return ERR_OTHER;
	}
	pbmih=(BITMAPINFOHEADER*)pbData;
	pbmih->biSize=sizeof(BITMAPINFOHEADER);
	pbmih->biWidth=pbmiSrc->bmiHeader.biWidth;
	pbmih->biHeight=pbmiSrc->bmiHeader.biHeight;
	pbmih->biPlanes=1;
	pbmih->biBitCount=pbmiSrc->bmiHeader.biBitCount;
	pbmih->biCompression=pbmiSrc->bmiHeader.biCompression==BI_RLE4
							|| pbmiSrc->bmiHeader.biCompression==BI_RLE8?
									BI_RGB:pbmiSrc->bmiHeader.biCompression;
	pbmih->biSizeImage=0;
	pbmih->biXPelsPerMeter=0;
	pbmih->biYPelsPerMeter=0;
	pbmih->biClrUsed=0;
	pbmih->biClrImportant=0;
	SrcInfoBytes=DIBGetInfoBytes(pData);
	if (pbmiSrc->bmiHeader.biBitCount<=8) {
		MemCopy(pbData+sizeof(BITMAPINFOHEADER),pbmiSrc->bmiColors,
										SrcInfoBytes-sizeof(BITMAPINFOHEADER));
	}
	pBits=(BYTE*)pbmiSrc+SrcInfoBytes;
	if (pbmiSrc->bmiHeader.biCompression==BI_RGB
			|| pbmiSrc->bmiHeader.biCompression==BI_BITFIELDS) {
		MemCopy(pbData+InfoBytes,pBits,BitsBytes);
	} else {
		DIBDecodeRLE(pbmiSrc,pBits,pbmiSrc->bmiHeader.biSizeImage,
															pbData+InfoBytes);
	}
	GlobalUnlock(hglobal);
	if (!OpenClipboard(hwnd)) {
		GlobalFree(hglobal);
		return ERR_OTHER;
	}
	if (!EmptyClipboard() || SetClipboardData(CF_DIB,hglobal)==NULL) {
		CloseClipboard();
		GlobalFree(hglobal);
		return ERR_OTHER;
	}
	CloseClipboard();
	return ERR_SUCCESS;
}


/*
	RLEkL
*/
BOOL DIBDecodeRLE(const BITMAPINFO *pbmi,const void *pSrcBits,size_t SrcSize,
																void *pDstBits)
{
	if (pbmi->bmiHeader.biCompression==BI_RLE4)
		return DIBDecodeRLE4(pbmi,pSrcBits,SrcSize,pDstBits);
	else if (pbmi->bmiHeader.biCompression==BI_RLE8)
		return DIBDecodeRLE8(pbmi,pSrcBits,SrcSize,pDstBits);
	return FALSE;
}


/*
	4rbgRLEkL
*/
BOOL DIBDecodeRLE4(const BITMAPINFO *pbmi,const void *pSrcBits,size_t SrcSize,
																void *pDstBits)
{
	UINT uBytesPerLine,uSrcBytes;
	const BYTE *p;
	BYTE *q;
	int x,y,c,i,nLength;

	uBytesPerLine=
			DIB_ROW_BYTES(pbmi->bmiHeader.biWidth,pbmi->bmiHeader.biBitCount);
	MemZero(pDstBits,uBytesPerLine*pbmi->bmiHeader.biHeight);
	uSrcBytes=min(pbmi->bmiHeader.biSizeImage,SrcSize);
	p=(BYTE*)pSrcBits;
	q=(BYTE*)pDstBits;
	x=y=0;
	while (uSrcBytes>0) {
		if (*p==0) {
			p++;
			if (--uSrcBytes==0)
				break;
			if (*p==0) {	/* End of line */
				if (++y==pbmi->bmiHeader.biHeight)
					break;
				x=0;
				p++;
				uSrcBytes--;
			} else if (*p==1) {	/* End of image */
				break;
			} else if (*p==2) {	/* Skip */
				if (uSrcBytes<3)
					break;
				uSrcBytes-=3;
				p++;
				x+=*p++;
				y+=*p++;
				if (x>pbmi->bmiHeader.biWidth || y>pbmi->bmiHeader.biHeight)
					break;
			} else {
				nLength=*p++;
				uSrcBytes--;
				if (x+nLength>pbmi->bmiHeader.biWidth
									|| uSrcBytes<(unsigned)((nLength+3)/4*2))
					break;
				for (i=0;i<nLength;i++) {
					if (i%2==0)
						c=*p&0xF0;
					else
						c=*p++<<4;
					q[y*uBytesPerLine+x/2]|=c>>(x%2*4);
					x++;
				}
				if (nLength%4==1)
					p++;
				if (nLength%4!=0)
					p++;
				uSrcBytes-=(nLength+3)/4*2;
			}
		} else {
			nLength=*p++;
			if (x+nLength>pbmi->bmiHeader.biWidth || uSrcBytes<2)
				break;
			for (i=0;i<nLength;i++) {
				q[y*uBytesPerLine+x/2]|=((*p<<(i%2*4))&0xF0)>>(x%2*4);
				x++;
			}
			p++;
			uSrcBytes-=2;
		}
	}
	return TRUE;
}


/*
	8rbgRLEkL
*/
BOOL DIBDecodeRLE8(const BITMAPINFO *pbmi,const void *pSrcBits,size_t SrcSize,
																void *pDstBits)
{
	UINT uBytesPerLine,uSrcBytes;
	const BYTE *p;
	BYTE *q;
	int x,y,i,nLength;

	uBytesPerLine=
			DIB_ROW_BYTES(pbmi->bmiHeader.biWidth,pbmi->bmiHeader.biBitCount);
	MemZero(pDstBits,uBytesPerLine*pbmi->bmiHeader.biHeight);
	uSrcBytes=min(pbmi->bmiHeader.biSizeImage,SrcSize);
	p=(BYTE*)pSrcBits;
	q=(BYTE*)pDstBits;
	x=y=0;
	while (uSrcBytes>0) {
		if (*p==0) {
			p++;
			if (--uSrcBytes==0)
				break;
			if (*p==0) {	/* End of line */
				if (++y==pbmi->bmiHeader.biHeight)
					break;
				if (x<(int)uBytesPerLine)
					q+=uBytesPerLine-x;
				x=0;
				p++;
				uSrcBytes--;
			} else if (*p==1) {	/* End of image */
				break;
			} else if (*p==2) {	/* Skip */
				if (uSrcBytes<3)
					break;
				uSrcBytes-=3;
				p++;
				x+=p[0];
				y+=p[2];
				if (x>pbmi->bmiHeader.biWidth || y>pbmi->bmiHeader.biHeight)
					break;
				q+=p[1]*uBytesPerLine+p[0];
				p+=2;
			} else {
				nLength=*p++;
				x+=nLength;
				uSrcBytes--;
				if (x>pbmi->bmiHeader.biWidth
									|| uSrcBytes<(unsigned)(nLength+nLength%2))
					break;
				for (i=0;i<nLength;i++)
					*q++=*p++;
				p+=nLength%2;
				uSrcBytes-=nLength+nLength%2;
			}
		} else {
			nLength=*p++;
			x+=nLength;
			uSrcBytes--;
			if (uSrcBytes==0 || x>pbmi->bmiHeader.biWidth)
				break;
			for (i=0;i<nLength;i++)
				*q++=*p;
			p++;
			uSrcBytes--;
		}
	}
	return TRUE;
}


/*
	BITMAPINFO ̃TCY߂
*/
UINT DIBGetInfoBytes(const void *pData)
{
	const BITMAPFILEHEADER *pbmfh;
	const BITMAPINFOHEADER *pbmih;
	UINT uInfoBytes;

	if (((BYTE*)pData)[0]=='B' && ((BYTE*)pData)[1]=='M') {
		pbmfh=(BITMAPFILEHEADER*)pData;
		pbmih=(BITMAPINFOHEADER*)((BYTE*)pData+sizeof(BITMAPFILEHEADER));
	} else {
		pbmfh=NULL;
		pbmih=(BITMAPINFOHEADER*)pData;
	}
	uInfoBytes=sizeof(BITMAPINFOHEADER);
	if (pbmih->biBitCount<=8) {
		if (pbmfh==NULL || pbmih->biClrUsed==0
				|| pbmih->biClrUsed==(unsigned)(1<<pbmih->biBitCount)
				|| pbmfh->bfOffBits==sizeof(BITMAPFILEHEADER)+
													sizeof(BITMAPINFOHEADER)+
										(1<<pbmih->biBitCount)*sizeof(RGBQUAD))
			uInfoBytes+=(1<<pbmih->biBitCount)*sizeof(RGBQUAD);
		else
			/* biClrUsed ̐pbg̐F݂Ȃt@C */
			uInfoBytes+=pbmih->biClrUsed*sizeof(RGBQUAD);
	} else if (pbmfh!=NULL && pbmih->biBitCount>=16 && pbmih->biClrUsed!=0
			&& pbmfh->bfOffBits==sizeof(BITMAPFILEHEADER)+
					sizeof(BITMAPINFOHEADER)+pbmih->biClrUsed*sizeof(RGBQUAD))
		uInfoBytes+=pbmih->biClrUsed*sizeof(RGBQUAD);
	if (pbmih->biCompression==BI_BITFIELDS)
		uInfoBytes+=3*sizeof(DWORD);
	return uInfoBytes;
}


UINT DIBCalcInfoBytes(const BITMAPINFOHEADER *pbmih)
{
	UINT uInfoBytes;

	uInfoBytes=sizeof(BITMAPINFOHEADER);
	if (pbmih->biBitCount<=8)
		uInfoBytes+=(1<<pbmih->biBitCount)*sizeof(RGBQUAD);
	else if (pbmih->biCompression==BI_BITFIELDS)
		uInfoBytes+=3*sizeof(DWORD);
	return uInfoBytes;
}


UINT DIBCalcBitsBytes(const BITMAPINFOHEADER *pbmih)
{
	return DIB_ROW_BYTES(pbmih->biWidth,pbmih->biBitCount)*abs(pbmih->biHeight);
}


UINT DIBCalcBytes(const BITMAPINFOHEADER *pbmih)
{
	return DIBCalcInfoBytes(pbmih)+DIBCalcBitsBytes(pbmih);
}


void DIBGetImageInfo(const void *pData,size_t DataSize,ImageInfo *pInfo)
{
	const BITMAPINFOHEADER *pbmih;

	if (((BYTE*)pData)[0]=='B' && ((BYTE*)pData)[1]=='M')
		pbmih=(const BITMAPINFOHEADER*)((BYTE*)pData+sizeof(BITMAPFILEHEADER));
	else
		pbmih=(const BITMAPINFOHEADER*)pData;
	pInfo->nWidth=pbmih->biWidth;
	pInfo->nHeight=pbmih->biHeight;
	pInfo->nBitsPerPixel=pbmih->biBitCount;
}




/******************************************************************************

	JPEG֐

******************************************************************************/


BITMAPINFO *JPEGConvertToDIB(const void *pSrcData,size_t DataSize)
{
	BITMAPINFOHEADER bmih;
	size_t InfoBytes,BitsBytes;
	BYTE *pBuffer;
	BITMAPINFO *pbmi;

	JPEGGetDIBInfoHeader(pSrcData,DataSize,&bmih);
	InfoBytes=DIBCalcInfoBytes(&bmih);
	BitsBytes=DIBCalcBitsBytes(&bmih);
	pBuffer=(BYTE*)MemAlloc(InfoBytes+BitsBytes);
	if (pBuffer==NULL)
		return NULL;
	pbmi=(BITMAPINFO*)pBuffer;
	pbmi->bmiHeader=bmih;
	if (pbmi->bmiHeader.biBitCount==8) {
		int i;

		for (i=0;i<256;i++) {
			pbmi->bmiColors[i].rgbBlue=i;
			pbmi->bmiColors[i].rgbGreen=i;
			pbmi->bmiColors[i].rgbRed=i;
			pbmi->bmiColors[i].rgbReserved=0;
		}
	}
	if (!JPEGDecode(pSrcData,DataSize,pBuffer+InfoBytes)) {
		MemFree(pBuffer);
		return NULL;
	}
	return pbmi;
}


BOOL JPEGGetDIBInfoHeader(const void *pData,size_t DataSize,
													BITMAPINFOHEADER *pbmih)
{
	ImageInfo Info;

	JPEGGetImageInfo(pData,DataSize,&Info);
	pbmih->biSize=sizeof(BITMAPINFOHEADER);
	pbmih->biWidth=Info.nWidth;
	pbmih->biHeight=Info.nHeight;
	pbmih->biPlanes=1;
	pbmih->biBitCount=Info.nBitsPerPixel;
	pbmih->biCompression=BI_RGB;
	pbmih->biSizeImage=0;
	pbmih->biXPelsPerMeter=0;
	pbmih->biYPelsPerMeter=0;
	pbmih->biClrUsed=0;
	pbmih->biClrImportant=0;
	return TRUE;
}


void JPEGGetImageInfo(const void *pData,size_t DataSize,ImageInfo *pInfo)
{
	const BYTE *p;

	p=(const BYTE*)pData;
	p+=2;	/* SOI */
	while (TRUE) {
		if (p[1]==JPEG_MARKER_SOF0 || p[1]==JPEG_MARKER_SOF2)
			break;
		p+=2;
		p+=(p[0]<<8)|p[1];
	}
	p+=2+2;
	pInfo->nHeight=(p[1]<<8)|p[2];
	pInfo->nWidth=(p[3]<<8)|p[4];
	pInfo->nBitsPerPixel=p[5]*8;
}




/******************************************************************************

	PNG֐

******************************************************************************/


ErrorType PNGCopyToClipboard(HWND hwnd,const void *pData,size_t DataSize)
{
	BITMAPINFO *pbmi=PNGConvertToDIB(pData,DataSize);
	ErrorType Err;

	if (pbmi==NULL)
		return ERR_OTHER;
	Err=DIBCopyToClipboard(hwnd,pbmi,DIBCalcBytes(&pbmi->bmiHeader));
	MemFree(pbmi);
	return Err;
}


BITMAPINFO *PNGConvertToDIB(const void *pData,size_t DataSize)
{
	BITMAPINFOHEADER bmih;
	size_t InfoBytes,BitsBytes;
	BYTE *pBuffer;

	PNGGetDIBInfoHeader(pData,&bmih);
	InfoBytes=DIBCalcInfoBytes(&bmih);
	BitsBytes=DIBCalcBitsBytes(&bmih);
	pBuffer=(BYTE*)MemAlloc(InfoBytes+BitsBytes);
	if (pBuffer==NULL)
		return NULL;
	MemCopy(pBuffer,&bmih,sizeof(BITMAPINFOHEADER));
	if (!PNGDecode(pData,DataSize,(BITMAPINFO*)pBuffer,pBuffer+InfoBytes)) {
		MemFree(pBuffer);
		return NULL;
	}
	return (BITMAPINFO*)pBuffer;
}


void PNGGetDIBInfoHeader(const void *pData,BITMAPINFOHEADER *pbmih)
{
	const BYTE *p;
	int BitsPerPlane;

	p=(const BYTE*)pData+8+4+4;
	pbmih->biSize=sizeof(BITMAPINFOHEADER);
	pbmih->biWidth=MSBFirst32(p);
	pbmih->biHeight=MSBFirst32(p+4);
	pbmih->biPlanes=1;
	BitsPerPlane=p[8];
	switch (p[9]) {
	case 0:	/* Grayscale */
	case 3:	/* Indexed */
		pbmih->biBitCount=BitsPerPlane==1?1:BitsPerPlane<=4?4:8;
		break;
	case 2:	/* RGB */
		pbmih->biBitCount=24;
		break;
	case 4:	/* Grayscale + Alpha */
	case 6:	/* RGB + Alpha */
		pbmih->biBitCount=32;
		break;
	}
	pbmih->biCompression=BI_RGB;
	pbmih->biSizeImage=0;
	pbmih->biXPelsPerMeter=0;
	pbmih->biYPelsPerMeter=0;
	pbmih->biClrUsed=0;
	pbmih->biClrImportant=0;
}


void PNGGetImageInfo(const void *pData,size_t DataSize,ImageInfo *pInfo)
{
	const BYTE *p;

	p=(const BYTE*)pData+8+4+4;
	pInfo->nWidth=MSBFirst32(p);
	pInfo->nHeight=MSBFirst32(p+4);
	pInfo->nBitsPerPixel=p[8];
	if (p[9]==2)
		pInfo->nBitsPerPixel*=3;
	else if (p[9]==4)
		pInfo->nBitsPerPixel*=2;
	else if (p[9]==6)
		pInfo->nBitsPerPixel*=4;
}




/******************************************************************************

	MAG֐

******************************************************************************/


ErrorType MAGSaveToFile(LPCTSTR pszFileName,const void *pData,size_t DataSize)
{
	HANDLE hFile;
	const BYTE *p;
	const MagHeader *pmh;
	DWORD Size,Write;

	hFile=CreateFile(pszFileName,GENERIC_WRITE,0,NULL,CREATE_ALWAYS,
												FILE_ATTRIBUTE_NORMAL,NULL);
	if (hFile==INVALID_HANDLE_VALUE)
		return ERR_FILEOPEN;
	p=(const BYTE*)pData+8;
	while (*p++!=0x1A);
	pmh=(MagHeader*)p;
	Size=(p-(BYTE*)pData)+pmh->dwPixelOffset+pmh->dwPixelSize;
	if (!WriteFile(hFile,pData,Size,&Write,NULL) || Write!=Size) {
		CloseHandle(hFile);
		return ERR_FILEWRITE;
	}
	if (!CloseHandle(hFile))
		return ERR_FILECLOSE;
	return ERR_SUCCESS;
}


static BOOL MAGDecode(const MagHeader *pmh,const BITMAPINFO *pbmiDst,
																void *pDstBits)
{
	int nLength=
			(pbmiDst->bmiHeader.biWidth*pbmiDst->bmiHeader.biBitCount+15)/16*2;
	UINT uDstRowBytes=DIB_ROW_BYTES(pbmiDst->bmiHeader.biWidth,
												pbmiDst->bmiHeader.biBitCount);
	BYTE *pbTemp,*p,*q;
	int x,y,i,j;
	DWORD FlagASize,FlagBSize,PixelSize;
	const BYTE *pbFlagA,*pbFlagB,*pbPixelData;
	BYTE bFlagA,bFlagABit;
	static const struct {
		int x,y;
	} Offset[16]= {
		{ 0, 0},{-2, 0},{-4, 0},{-8, 0},{ 0,-1},{-2,-1},{ 0,-2},{-2,-2},
		{-4,-2},{ 0,-4},{-2,-4},{-4,-4},{ 0,-8},{-2,-8},{-4,-8},{ 0,-16}
	};

	pbTemp=(BYTE*)MemAlloc(nLength/4);
	if (pbTemp==NULL)
		return FALSE;
	MemZero(pbTemp,nLength/4);
	FlagASize=pmh->dwFlagBOffset-pmh->dwFlagAOffset;
	FlagBSize=pmh->dwFlagBSize;
	PixelSize=pmh->dwPixelSize;
	pbFlagA=(const BYTE*)pmh+pmh->dwFlagAOffset;
	pbFlagB=(const BYTE*)pmh+pmh->dwFlagBOffset;
	pbPixelData=(const BYTE*)pmh+pmh->dwPixelOffset;
	bFlagABit=0;
	for (y=0;y<pbmiDst->bmiHeader.biHeight;y++) {
		p=pbTemp;
		for (i=0;i<nLength/4;i++) {
			if (bFlagABit==0) {
				if (FlagASize==0)
					goto OnError;
				bFlagABit=0x80;
				bFlagA=*pbFlagA++;
				FlagASize--;
			}
			if ((bFlagA&bFlagABit)!=0) {
				if (FlagBSize==0)
					goto OnError;
				*p^=*pbFlagB++;
				FlagBSize--;
			}
			p++;
			bFlagABit>>=1;
		}
		q=(BYTE*)pDstBits+(pbmiDst->bmiHeader.biHeight-1-y)*uDstRowBytes;
		for (x=0,i=0;x<nLength;x+=2,i++) {
			j=(pbTemp[i/2]>>(1-i%2)*4)&0x0F;
			if (j==0) {
				if (PixelSize<2)
					goto OnError;
				*q++=*pbPixelData++;
				*q++=*pbPixelData++;
				PixelSize-=2;
			} else {
				int x1,y1;

				x1=x+Offset[j].x;
				y1=y+Offset[j].y;
				if (x1<0 || y1<0)
					goto OnError;
				p=(BYTE*)pDstBits+
							(pbmiDst->bmiHeader.biHeight-1-y1)*uDstRowBytes+x1;
				*q++=*p++;
				*q++=*p;
			}
		}
		if ((pmh->bScreenMode&MAG_200LINE)!=0) {
			q=(BYTE*)pDstBits+(pbmiDst->bmiHeader.biHeight-1-y)*uDstRowBytes;
			MemCopy(q,q-uDstRowBytes,uDstRowBytes);
			y++;
		}
	}
	MemFree(pbTemp);
	return TRUE;

OnError:
	MemFree(pbTemp);
	return FALSE;
}


ErrorType MAGCopyToClipboard(HWND hwnd,const void *pData,size_t DataSize)
{
	BITMAPINFOHEADER bmih;
	UINT InfoBytes,BitsBytes;
	HGLOBAL hglobal;
	BYTE *pbClipboard;
	BITMAPINFO *pbmi;
	const BYTE *p;
	const MagHeader *pMagHeader;
	int i;

	MAGGetDIBInfoHeader(pData,DataSize,&bmih);
	InfoBytes=DIBCalcInfoBytes(&bmih);
	BitsBytes=DIBCalcBitsBytes(&bmih);
	hglobal=GlobalAlloc(GMEM_MOVEABLE | GMEM_SHARE,InfoBytes+BitsBytes);
	if (hglobal==NULL)
		return ERR_MEMORYALLOC;
	pbClipboard=(BYTE*)GlobalLock(hglobal);
	if (pbClipboard==NULL) {
		GlobalFree(hglobal);
		return ERR_OTHER;
	}
	pbmi=(BITMAPINFO*)pbClipboard;
	pbmi->bmiHeader=bmih;
	p=(const BYTE*)pData+8;	/* Skip "MAKI02  " */
	while (*p++!=0x1A);		/* Skip "User name" and "Comment" */
	pMagHeader=(const MagHeader*)p;
	p+=sizeof(MagHeader);
	for (i=0;i<1<<pbmi->bmiHeader.biBitCount;i++) {
		pbmi->bmiColors[i].rgbGreen=*p++;
		pbmi->bmiColors[i].rgbRed=*p++;
		pbmi->bmiColors[i].rgbBlue=*p++;
		pbmi->bmiColors[i].rgbReserved=0;
	}
	if (!MAGDecode(pMagHeader,pbmi,pbClipboard+InfoBytes)) {
		GlobalUnlock(hglobal);
		GlobalFree(hglobal);
		return ERR_OTHER;
	}
	GlobalUnlock(hglobal);
	if (!OpenClipboard(hwnd)) {
		GlobalFree(hglobal);
		return ERR_OTHER;
	}
	if (!EmptyClipboard() || SetClipboardData(CF_DIB,hglobal)==NULL) {
		CloseClipboard();
		GlobalFree(hglobal);
		return ERR_OTHER;
	}
	CloseClipboard();
	return ERR_SUCCESS;
}


BITMAPINFO *MAGConvertToDIB(const void *pData,size_t DataSize)
{
	BITMAPINFOHEADER bmih;
	size_t InfoBytes,BitsBytes;
	BYTE *pBuffer;
	const BYTE *p;
	const MagHeader *pmh;
	BITMAPINFO *pbmi;
	int i;

	MAGGetDIBInfoHeader(pData,DataSize,&bmih);
	InfoBytes=DIBCalcInfoBytes(&bmih);
	BitsBytes=DIBCalcBitsBytes(&bmih);
	pBuffer=(BYTE*)MemAlloc(InfoBytes+BitsBytes);
	if (pBuffer==NULL)
		return NULL;
	pbmi=(BITMAPINFO*)pBuffer;
	pbmi->bmiHeader=bmih;
	p=(const BYTE*)pData+8;	/* Skip "MAKI02  " */
	while (*p++!=0x1A);		/* Skip "User name" and "Comment" */
	pmh=(const MagHeader*)p;
	p+=sizeof(MagHeader);
	for (i=0;i<1<<pbmi->bmiHeader.biBitCount;i++) {
		pbmi->bmiColors[i].rgbGreen=*p++;
		pbmi->bmiColors[i].rgbRed=*p++;
		pbmi->bmiColors[i].rgbBlue=*p++;
		pbmi->bmiColors[i].rgbReserved=0;
	}
	if (!MAGDecode(pmh,pbmi,pBuffer+InfoBytes)) {
		MemFree(pBuffer);
		return NULL;
	}
	return pbmi;
}


BOOL MAGGetDIBInfoHeader(const void *pData,size_t DataSize,
													BITMAPINFOHEADER *pbmih)
{
	const BYTE *p;
	const MagHeader *pmh;

	p=(BYTE*)pData+8;	/* Skip "MAKI02  " */
	while (*p++!=0x1A);	/* Skip "User name" and "Comment" */
	pmh=(MagHeader*)p;
	pbmih->biSize=sizeof(BITMAPINFOHEADER);
	pbmih->biWidth=pmh->wXEnd-pmh->wXStart+1;
	pbmih->biHeight=pmh->wYEnd-pmh->wYStart+1;
	if ((pmh->bScreenMode&MAG_200LINE)!=0)
		pbmih->biHeight*=2;
	pbmih->biPlanes=1;
	pbmih->biBitCount=(pmh->bScreenMode&MAG_256COLORS)!=0?8:4;
	pbmih->biCompression=BI_RGB;
	pbmih->biSizeImage=0;
	pbmih->biXPelsPerMeter=0;
	pbmih->biYPelsPerMeter=0;
	pbmih->biClrUsed=0;
	pbmih->biClrImportant=0;
	return TRUE;
}


void MAGGetImageInfo(const void *pData,size_t DataSize,ImageInfo *pInfo)
{
	const BYTE *p;
	const MagHeader *pmh;

	p=(BYTE*)pData+8;	/* Skip "MAKI02  " */
	while (*p++!=0x1A);	/* Skip "User name" and "Comment" */
	pmh=(MagHeader*)p;
	pInfo->nWidth=pmh->wXEnd-pmh->wXStart+1;
	pInfo->nHeight=pmh->wYEnd-pmh->wYStart+1;
	if ((pmh->bScreenMode&MAG_200LINE)!=0)
		pInfo->nHeight*=2;
	pInfo->nBitsPerPixel=(pmh->bScreenMode&MAG_256COLORS)!=0?8:4;
}




/******************************************************************************

	ICON֐

******************************************************************************/


#define IsIconDir(p) (*(const BYTE*)(p)==0)


static const BITMAPINFOHEADER *FindMaxSizeIcon(const void *pData)
{
	const ICONDIR *pid=(const ICONDIR*)pData;
	const ICONDIRENTRY *pide=
						(const ICONDIRENTRY*)((BYTE*)pData+sizeof(ICONDIR));
	int i;
	int MaxSize,MaxBitsPerPixel;
	const BITMAPINFOHEADER *pbmih,*pbmihMax;

	MaxSize=MaxBitsPerPixel=0;
	for (i=0;i<pid->idCount;i++) {
		pbmih=(const BITMAPINFOHEADER*)((BYTE*)pData+pide[i].dwImageOffset);
		if (pbmih->biWidth>MaxSize
				|| (pbmih->biWidth==MaxSize
									&& pbmih->biBitCount>MaxBitsPerPixel)) {
			MaxSize=pbmih->biWidth;
			MaxBitsPerPixel=pbmih->biBitCount;
			pbmihMax=pbmih;
		}
	}
	return pbmihMax;
}


static const BITMAPINFOHEADER *GetIconBitmapInfoHeader(const void *pData)
{
	if (IsIconDir(pData))
		return FindMaxSizeIcon(pData);
	return (const BITMAPINFOHEADER*)pData;
}


ErrorType IconSaveToFile(LPCTSTR pszFileName,const void *pData,size_t DataSize)
{
	if (IsIconDir(pData))
		return DataSaveToFile(pszFileName,pData,DataSize);
	{
		HANDLE hFile;
		DWORD dwWrite;
		const BITMAPINFO *pbmi=(const BITMAPINFO*)pData;
		ICONDIR id;
		ICONDIRENTRY ide;

		hFile=CreateFile(pszFileName,GENERIC_WRITE,0,NULL,CREATE_ALWAYS,
												FILE_ATTRIBUTE_NORMAL,NULL);
		if (hFile==INVALID_HANDLE_VALUE)
			return ERR_FILEOPEN;
		id.idReserved=0;
		id.idType=1;
		id.idCount=1;
		if (!WriteFile(hFile,&id,sizeof(ICONDIR),&dwWrite,NULL)
												|| dwWrite!=sizeof(ICONDIR)) {
			CloseHandle(hFile);
			return ERR_FILEWRITE;
		}
		ide.bWidth=(BYTE)pbmi->bmiHeader.biWidth;
		ide.bHeight=(BYTE)pbmi->bmiHeader.biHeight/2;
		ide.bColorCount=pbmi->bmiHeader.biBitCount<=4?
											1<<pbmi->bmiHeader.biBitCount:0;
		ide.bReserved=0;
		ide.wPlanes=1;
		ide.wBitCount=pbmi->bmiHeader.biBitCount;
		ide.dwBytesInRes=DataSize;
		ide.dwImageOffset=sizeof(ICONDIR)+sizeof(ICONDIRENTRY);
		if (!WriteFile(hFile,&ide,sizeof(ICONDIRENTRY),&dwWrite,NULL)
											|| dwWrite!=sizeof(ICONDIRENTRY)
				|| !WriteFile(hFile,pData,DataSize,&dwWrite,NULL)
														|| dwWrite!=DataSize) {
			CloseHandle(hFile);
			return ERR_FILEWRITE;
		}
		if (!CloseHandle(hFile))
			return ERR_FILECLOSE;
		return ERR_SUCCESS;
	}
}


ErrorType IconCopyToClipboard(HWND hwnd,const void *pData,size_t DataSize)
{
	const BITMAPINFOHEADER *pbmih;
	unsigned int InfoSize,PalSize,BitsSize;
	void *pBuffer;
	BYTE *p;
	ErrorType Err;

	pbmih=GetIconBitmapInfoHeader(pData);
	InfoSize=sizeof(BITMAPINFOHEADER);
	if (pbmih->biBitCount<=8)
		PalSize=(1<<pbmih->biBitCount)*sizeof(RGBQUAD);
	InfoSize+=PalSize;
	BitsSize=DIB_ROW_BYTES(pbmih->biWidth,pbmih->biBitCount)*
														(pbmih->biHeight/2);
	pBuffer=MemAlloc(InfoSize+BitsSize);
	if (pBuffer==NULL)
		return ERR_MEMORYALLOC;
	p=(BYTE*)pBuffer;
	IconGetDIBInfoHeader(pData,DataSize,(BITMAPINFOHEADER*)p);
	p+=sizeof(BITMAPINFOHEADER);
	if (pbmih->biBitCount<=8) {
		MemCopy(p,pbmih+1,PalSize);
		p+=PalSize;
	}
	MemCopy(p,(BYTE*)pbmih+InfoSize,BitsSize);
	Err=DIBCopyToClipboard(hwnd,pBuffer,InfoSize+BitsSize);
	MemFree(pBuffer);
	return Err;
}


BITMAPINFO *IconConvertToDIB(const void *pData,COLORREF crBgColor)
{
	BITMAPINFO *pbmi;
	const BITMAPINFOHEADER *pbmihIcon;
	unsigned int InfoSize,PalSize,BitsSize;

	pbmihIcon=GetIconBitmapInfoHeader(pData);
	InfoSize=sizeof(BITMAPINFOHEADER);
	if (pbmihIcon->biBitCount<=8) {
		PalSize=(1<<pbmihIcon->biBitCount)*sizeof(RGBQUAD);
		InfoSize+=PalSize;
	}
	BitsSize=DIB_ROW_BYTES(pbmihIcon->biWidth,pbmihIcon->biBitCount)*
													(pbmihIcon->biHeight/2);
	pbmi=(BITMAPINFO*)MemAlloc(InfoSize+BitsSize);
	if (pbmi==NULL)
		return NULL;
	pbmi->bmiHeader.biSize=sizeof(BITMAPINFOHEADER);
	pbmi->bmiHeader.biWidth=pbmihIcon->biWidth;
	pbmi->bmiHeader.biHeight=pbmihIcon->biHeight/2;
	pbmi->bmiHeader.biPlanes=1;
	pbmi->bmiHeader.biBitCount=pbmihIcon->biBitCount;
	pbmi->bmiHeader.biCompression=BI_RGB;
	pbmi->bmiHeader.biSizeImage=0;
	pbmi->bmiHeader.biXPelsPerMeter=0;
	pbmi->bmiHeader.biYPelsPerMeter=0;
	pbmi->bmiHeader.biClrUsed=0;
	pbmi->bmiHeader.biClrImportant=0;
	if (pbmihIcon->biBitCount<=8)
		MemCopy(pbmi->bmiColors,pbmihIcon+1,PalSize);
	MemCopy((BYTE*)pbmi+InfoSize,(BYTE*)pbmihIcon+InfoSize,BitsSize);
	return pbmi;
}


BOOL IconGetDIBInfoHeader(const void *pData,size_t DataSize,
													BITMAPINFOHEADER *pbmih)
{
	ImageInfo Info;

	IconGetImageInfo(pData,DataSize,&Info);
	pbmih->biSize=sizeof(BITMAPINFOHEADER);
	pbmih->biWidth=Info.nWidth;
	pbmih->biHeight=Info.nHeight;
	pbmih->biPlanes=1;
	pbmih->biBitCount=Info.nBitsPerPixel;
	pbmih->biCompression=BI_RGB;
	pbmih->biSizeImage=0;
	pbmih->biXPelsPerMeter=0;
	pbmih->biYPelsPerMeter=0;
	pbmih->biClrUsed=0;
	pbmih->biClrImportant=0;
	return TRUE;
}


void IconGetImageInfo(const void *pData,size_t DataSize,ImageInfo *pInfo)
{
	const BITMAPINFOHEADER *pbmih;

	pbmih=GetIconBitmapInfoHeader(pData);
	pInfo->nWidth=pbmih->biWidth;
	pInfo->nHeight=pbmih->biHeight/2;
	pInfo->nBitsPerPixel=pbmih->biBitCount;
}




/******************************************************************************

	ANI֐

******************************************************************************/


static const void *FindANICursor(const void *pData)
{
	const BYTE *p;
	unsigned int Size;

	p=(const BYTE*)pData+(4+4+4+4);
	p+=4+(LSBFirst32(p)+1)/2*2;
	while (TRUE) {
		if (p[0]=='L' && p[1]=='I' && p[2]=='S' && p[3]=='T') {
			p+=4+4+4+4+4;
			break;
		} else {
			p+=4;
			Size=(LSBFirst32(p)+1)/2*2;
			p+=4+Size;
		}
	}
	return p;
}


ErrorType ANICopyToClipboard(HWND hwnd,const void *pData,size_t DataSize)
{
	return IconCopyToClipboard(hwnd,FindANICursor(pData),DataSize);
}


BITMAPINFO *ANIConvertToDIB(const void *pData,COLORREF crBgColor)
{
	return IconConvertToDIB(FindANICursor(pData),crBgColor);
}


BOOL ANIGetDIBInfoHeader(const void *pData,size_t DataSize,
													BITMAPINFOHEADER *pbmih)
{
	const BYTE *p=(const BYTE*)FindANICursor(pData);

	return IconGetDIBInfoHeader(p,DataSize-(p-(const BYTE*)pData),pbmih);
}


void ANIGetImageInfo(const void *pData,size_t DataSize,ImageInfo *pInfo)
{
	IconGetImageInfo(FindANICursor(pData),DataSize,pInfo);
}




/******************************************************************************

	WAVE֐

******************************************************************************/


/*
	WAVENbv{[hɃRs[
*/
ErrorType WAVECopyToClipboard(HWND hwnd,const void *pData,size_t DataSize)
{
	DWORD dwDataSize;
	HGLOBAL hglobal;
	void *p;

	dwDataSize=*(DWORD*)((BYTE*)pData+4)+8;
	hglobal=GlobalAlloc(GMEM_MOVEABLE | GMEM_SHARE,dwDataSize);
	if (hglobal==NULL)
		return ERR_MEMORYALLOC;
	p=GlobalLock(hglobal);
	if (p==NULL) {
		GlobalFree(hglobal);
		return ERR_OTHER;
	}
	MemCopy(p,pData,dwDataSize);
	GlobalUnlock(hglobal);
	if (!OpenClipboard(hwnd)) {
		GlobalFree(hglobal);
		return ERR_OTHER;
	}
	if (!EmptyClipboard() || SetClipboardData(CF_WAVE,hglobal)==NULL) {
		CloseClipboard();
		GlobalFree(hglobal);
		return ERR_OTHER;
	}
	CloseClipboard();
	return ERR_SUCCESS;
}


void WAVEGetSoundInfo(const void *pData,size_t DataSize,SoundInfo *pInfo)
{
	const WAVEFORMATEX *pwfx=(const WAVEFORMATEX*)((BYTE*)pData+8+4+8);

	pInfo->nChannels=pwfx->nChannels;
	pInfo->nBitsPerSample=pwfx->wBitsPerSample;
	pInfo->nSamplesPerSec=pwfx->nSamplesPerSec;
	if (pwfx->wFormatTag==WAVE_FORMAT_PCM) {
		const BYTE *pbDataChunk;
		DWORD dwDataSize;

		pbDataChunk=(BYTE*)pData+8+4+8+*(DWORD*)((BYTE*)pData+8+4+4);
		if (pbDataChunk[0]=='f')	/* "fact" */
			pbDataChunk+=12;
		dwDataSize=*(DWORD*)(pbDataChunk+4);
		pInfo->uLength=dwDataSize*8/pwfx->wBitsPerSample*1000/
														pwfx->nSamplesPerSec;
	} else {
		DWORD dwFact;

		dwFact=*(DWORD*)((BYTE*)pwfx+sizeof(WAVEFORMATEX)+pwfx->cbSize+8);
		pInfo->uLength=dwFact*1000/pwfx->nSamplesPerSec;
	}
}




/******************************************************************************

	MIDI֐

******************************************************************************/




/******************************************************************************

	MP3֐

******************************************************************************/


void MP3GetSoundInfo(const void *pData,size_t DataSize,SoundInfo *pInfo)
{
	const BYTE *p,*q;
	DWORD Size;
	MP3FrameInfo FrameInfo;
	DWORD NumFrames;

	p=(const BYTE*)pData;
	if (*p=='I') {
		/* ID3 v2 */
		p+=3+3;
		Size=(p[0]<<(7*3))|(p[1]<<(7*2))|(p[2]<<7)|p[3];
		p+=4+Size;
	}
	MP3GetFrameInfo(p,&FrameInfo);
	NumFrames=0;
	q=p+4;
	while (q<p+FrameInfo.FrameSize) {
		if ((q[0]=='X' && q[1]=='i' && q[2]=='n' && q[3]=='g')
				|| (q[0]=='I' && q[1]=='n' && q[2]=='f' && q[3]=='o')) {
			q+=4;
			if ((q[3]&0x03)==0x03) {
				q+=4;
				NumFrames=MSBFirst32(q);
				//q+=4;
				//Bytes=MSBFirst32(q);
				//q+=4;
			}
			break;
		} else if (q[0]=='V' && q[1]=='B' && q[2]=='R' && q[3]=='I') {
			q+=4+2+2+2;
			//Bytes=MSBFirst32(q);
			q+=4;
			NumFrames=MSBFirst32(q);
			//q+=4;
			break;
		}
		q++;
	}
	if (NumFrames==0) {
		size_t RemainSize=DataSize-(q-p);
		while (RemainSize>4) {
			MP3FrameInfo CurFrame;

			if (q[0]!=0xFF || (q[1]&0xE0)!=0xE0
					|| !MP3GetFrameInfo(q,&CurFrame)
					|| RemainSize<CurFrame.FrameSize)
				break;
			q+=CurFrame.FrameSize;
			RemainSize-=CurFrame.FrameSize;
			NumFrames++;
		}
	}
	pInfo->nChannels=FrameInfo.ChannelMode==MPEG_CHANNEL_MONO?1:2;
	pInfo->nBitsPerSample=16;
	pInfo->nSamplesPerSec=FrameInfo.Frequency;
	pInfo->uLength=MulDiv(NumFrames,FrameInfo.SamplesPerFrame*1000,
														FrameInfo.Frequency);
}


BOOL MP3GetFrameInfo(const void *pData,MP3FrameInfo *pInfo)
{
	const BYTE *p;
	BYTE Version,Layer;
	int BitRateIndex,FrequencyIndex;
	WORD BitRate;
	static const WORD BitRateTable1[3][14] = {
		{ 32, 40, 48, 56, 64, 80, 96,112,128,160,192,224,256,320},
		{ 32, 48, 56, 64, 80, 96,112,128,160,192,224,256,320,384},
		{ 32, 64, 96,128,160,192,224,256,288,320,352,384,416,448},
	};
	static const WORD BitRateTable2[3][14] = {
		{  8, 16, 24, 32, 40, 48, 56, 64, 80, 96,112,128,144,160},
		{  8, 16, 24, 32, 40, 48, 56, 64, 80, 96,112,128,144,160},
		{ 32, 48, 56, 64, 80, 96,112,128,144,160,176,192,224,256},
	};
	static const WORD FrequencyTable[4][4] = {
		{11025,12000, 8000},	// MPEG 2.5
		{    0,    0,    0},	// Reserved
		{22050,24000,16000},	// MPEG 2
		{44100,48000,32000},	// MPEG 1
	};
	static const WORD SamplesPerFrameTable[4][3] = {
		{ 576,1152, 384},	// MPEG 2.5
		{   0,   0,   0},	// Reserved
		{ 576,1152, 384},	// MPEG 2
		{1152,1152, 384},	// MPEG 1
	};

	p=(const BYTE*)pData;
	if (p[0]!=0xFF || (p[1]&0xE0)!=0xE0)
		return FALSE;
	Version=(p[1]&0x18)>>3;
	Layer=(p[1]&0x06)>>1;
	if (Version==1 || Layer==0)
		return FALSE;
	pInfo->Version=Version;
	pInfo->Layer=Layer;
	pInfo->SamplesPerFrame=SamplesPerFrameTable[Version][Layer-1];
	BitRateIndex=p[2]>>4;
	FrequencyIndex=(p[2]&0x0C)>>2;
	if (BitRateIndex==0x0 || BitRateIndex==0xF || FrequencyIndex==0x3)
		return FALSE;
	if (Version==3)
		BitRate=BitRateTable1[Layer-1][BitRateIndex-1];
	else
		BitRate=BitRateTable2[Layer-1][BitRateIndex-1];
	pInfo->BitRate=(DWORD)BitRate*1000;
	pInfo->Frequency=FrequencyTable[Version][FrequencyIndex];
	pInfo->Padding=(p[2]&0x02)>>1;
	pInfo->ChannelMode=(p[3]&0xC0)>>6;
	pInfo->FrameSize=(WORD)((pInfo->SamplesPerFrame/8*pInfo->BitRate)/
											pInfo->Frequency+pInfo->Padding);
	return TRUE;
}




/******************************************************************************

	AVI֐

******************************************************************************/


#define NOCOMPMAN
#include <vfw.h>


void AVIGetVideoInfo(const void *pData,size_t DataSize,VideoInfo *pInfo)
{
	DWORD dwMainHeaderSize;
	const MainAVIHeader *pmavih;
	const AVIStreamHeader *pavish;

	pmavih=(const MainAVIHeader*)((BYTE*)pData+12+12+8);
	dwMainHeaderSize=*(DWORD*)((BYTE*)pData+12+12+4);
	pavish=(const AVIStreamHeader*)((BYTE*)pData+12+12+8+dwMainHeaderSize+12+8);
	pInfo->nWidth=pmavih->dwWidth;
	pInfo->nHeight=pmavih->dwHeight;
	pInfo->uLength=pavish->dwLength*1000/(pavish->dwRate/pavish->dwScale);
}
