/******************************************************************************
*                                                                             *
*    Preview.c                              Copyright(c) 2005-2010 itow,y.    *
*                                                                             *
******************************************************************************/

/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <windows.h>
#include <commctrl.h>
#define NOCOMPMAN
#include <vfw.h>
#include <tchar.h>
#include "DataCutter.h"
#include "Preview.h"
#include "Formats.h"
#include "FileUtil.h"
#include "Util.h"
#include "resource.h"

#pragma comment(lib, "winmm.lib")
#pragma comment(lib, "vfw32.lib")


extern HINSTANCE hInst;


#define PREVIEW_CANVAS_WINDOW_CLASS TEXT("DataCutter Preview Canvas")

#define ZOOM_MIN	10
#define ZOOM_MAX	1000

typedef enum {
	PREVIEW_FORMAT_UNDEFINED,
	PREVIEW_FORMAT_IMAGE,
	PREVIEW_FORMAT_SOUND,
	PREVIEW_FORMAT_MIDI,
	PREVIEW_FORMAT_MP3,
	PREVIEW_FORMAT_VIDEO
} PreviewFormatType;

typedef struct {
	HWND hwndToolbar;
	HWND hwndCanvas;
	PreviewFormatType Format;
	union {
		struct {
			HBITMAP hbm;
			void *pBits;
			int nScrollLeft;
			int nScrollTop;
			BOOL fScrollBarSetting;
		} Image;
		struct {
			WAVEFORMATEX *pwfx;
			void *pData;
			size_t DataSize;
			HWAVEOUT hwo;
			WAVEHDR wh;
			BOOL fPause;
		} Sound;
		struct {
			const void *pData;
			size_t DataSize;
			BOOL fOpened;
			TCHAR szTempFileName[MAX_PATH];
			BOOL fPause;
		} Midi;
		struct {
			const void *pData;
			size_t DataSize;
			BOOL fOpened;
			TCHAR szTempFileName[MAX_PATH];
			BOOL fPause;
		} MP3;
		struct {
			const void *pData;
			size_t DataSize;
			BOOL fOpened;
			TCHAR szTempFilePath[MAX_PATH];
			HWND hwndSeekBar;
			HWND hwndMCI;
		} Video;
	} Data;
	BOOL fFitImageToWindow;
	int nZoom;
	int nZoomStep;
	BOOL fShowTipHelp;
} PreviewWindowInfo;

#define GetPreviewWindowInfo(hwnd) \
	((PreviewWindowInfo*)GetWindowLongPtr(hwnd,0))




/*
	XN[o[NCAg̈̑傫Zo
*/
static void CalcClientRect(HWND hwnd,RECT *prc)
{
	GetWindowRect(hwnd,prc);
	OffsetRect(prc,-prc->left,-prc->top);
	if ((GetWindowLong(hwnd,GWL_STYLE)&WS_BORDER)!=0) {
		prc->right-=GetSystemMetrics(SM_CXBORDER)*2;
		prc->bottom-=GetSystemMetrics(SM_CYBORDER)*2;
	}
	if ((GetWindowLong(hwnd,GWL_EXSTYLE)&WS_EX_CLIENTEDGE)!=0) {
		prc->right-=GetSystemMetrics(SM_CXEDGE)*2;
		prc->bottom-=GetSystemMetrics(SM_CYEDGE)*2;
	}
}


/*
	XN[o[ݒ肷
*/
static void SetScrollBar(HWND hwnd,PreviewWindowInfo *pInfo)
{
	BITMAP bm;
	RECT rc;
	int HPage,VPage;
	BOOL fHScroll,fVScroll;
	SCROLLINFO si;

	GetObject(pInfo->Data.Image.hbm,sizeof(BITMAP),&bm);
	CalcClientRect(hwnd,&rc);
	HPage=(rc.right*100+(pInfo->nZoom-1))/pInfo->nZoom;
	if (HPage<bm.bmWidth) {
		fHScroll=TRUE;
		rc.bottom-=GetSystemMetrics(SM_CYHSCROLL);
	} else
		fHScroll=FALSE;
	VPage=(rc.bottom*100+(pInfo->nZoom-1))/pInfo->nZoom;
	if (VPage<bm.bmHeight) {
		fVScroll=TRUE;
		rc.right-=GetSystemMetrics(SM_CXVSCROLL);
		HPage=(rc.right*100+(pInfo->nZoom-1))/pInfo->nZoom;
		if (!fHScroll && HPage<bm.bmWidth) {
			rc.bottom-=GetSystemMetrics(SM_CYHSCROLL);
			VPage=(rc.bottom*100+(pInfo->nZoom-1))/pInfo->nZoom;
			fHScroll=TRUE;
		}
	} else
		fVScroll=FALSE;
	// XN[o[̐ݒ
	pInfo->Data.Image.fScrollBarSetting=TRUE;
	si.cbSize=sizeof(SCROLLINFO);
	si.fMask=SIF_PAGE | SIF_POS | SIF_RANGE;
	si.nMin=0;
	// XN[o[
	si.nMax=bm.bmWidth-1;
	si.nPage=HPage;
	if (HPage>=bm.bmWidth)
		si.nPos=0;
	else if (pInfo->Data.Image.nScrollLeft>bm.bmWidth-HPage)
		si.nPos=bm.bmWidth-HPage;
	else
		si.nPos=pInfo->Data.Image.nScrollLeft;
	SetScrollInfo(hwnd,SB_HORZ,&si,TRUE);
	ShowScrollBar(hwnd,SB_HORZ,fHScroll);
	pInfo->Data.Image.nScrollLeft=si.nPos;
	// cXN[o[
	si.nMax=bm.bmHeight-1;
	si.nPage=VPage;
	if (VPage>=bm.bmHeight)
		si.nPos=0;
	else if (pInfo->Data.Image.nScrollTop>bm.bmHeight-VPage)
		si.nPos=bm.bmHeight-VPage;
	else
		si.nPos=pInfo->Data.Image.nScrollTop;
	SetScrollInfo(hwnd,SB_VERT,&si,TRUE);
	ShowScrollBar(hwnd,SB_VERT,fVScroll);
	pInfo->Data.Image.nScrollTop=si.nPos;
	pInfo->Data.Image.fScrollBarSetting=FALSE;
}


/*
	摜XN[
*/
static void ScrollImage(HWND hwnd,PreviewWindowInfo *pInfo,int nHScroll,
											int nVScroll,BOOL fSetScrollBar)
{
	int nHScrollSize,nVScrollSize;
	RECT rc;

	nHScrollSize=-nHScroll*pInfo->nZoom/100;
	nVScrollSize=-nVScroll*pInfo->nZoom/100;
	GetClientRect(hwnd,&rc);
	if (pInfo->nZoom==100 && nHScrollSize<rc.right && nVScrollSize<rc.bottom)
		ScrollWindowEx(hwnd,nHScrollSize,nVScrollSize,NULL,NULL,NULL,NULL,
																SW_INVALIDATE);
	else
		/* {100%łȂꍇAŜĕ`悵ȂƂ */
		InvalidateRect(hwnd,NULL,FALSE);
	if (nHScroll!=0)
		pInfo->Data.Image.nScrollLeft+=nHScroll;
	if (nVScroll!=0)
		pInfo->Data.Image.nScrollTop+=nVScroll;
	if (fSetScrollBar) {
		SCROLLINFO si;

		si.cbSize=sizeof(SCROLLINFO);
		si.fMask=SIF_POS;
		if (nHScroll!=0) {
			si.nPos=pInfo->Data.Image.nScrollLeft;
			SetScrollInfo(hwnd,SB_HORZ,&si,TRUE);
		}
		if (nVScroll!=0) {
			si.nPos=pInfo->Data.Image.nScrollTop;
			SetScrollInfo(hwnd,SB_VERT,&si,TRUE);
		}
	}
}


/*
	LoXEBhE
*/
static LRESULT CALLBACK PreviewCanvasWndProc(HWND hwnd,UINT uMsg,WPARAM wParam,
																LPARAM lParam)
{
	switch (uMsg) {
	case WM_CREATE:
		ShowScrollBar(hwnd,SB_BOTH,FALSE);
		return 0;

	case WM_PAINT:
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(GetParent(hwnd));
			PAINTSTRUCT ps;
			HBRUSH hbrBackground;

			BeginPaint(hwnd,&ps);
			hbrBackground=CreateSolidBrush(GetSysColor(COLOR_APPWORKSPACE));
			switch (pInfo->Format) {
			case PREVIEW_FORMAT_IMAGE:
				if (pInfo->Data.Image.hbm!=NULL) {
					BITMAP bm;
					HDC hdcMemory;
					HBITMAP hbmOld;
					int sx,sy;
					RECT rc;

					GetObject(pInfo->Data.Image.hbm,sizeof(BITMAP),&bm);
					hdcMemory=CreateCompatibleDC(ps.hdc);
					hbmOld=(HBITMAP)SelectObject(hdcMemory,
														pInfo->Data.Image.hbm);
					if (pInfo->nZoom==100) {
						sx=bm.bmWidth-pInfo->Data.Image.nScrollLeft;
						sy=bm.bmHeight-pInfo->Data.Image.nScrollTop;
						if (ps.rcPaint.left<sx && ps.rcPaint.top<sy)
							BitBlt(ps.hdc,ps.rcPaint.left,ps.rcPaint.top,
									min(sx,ps.rcPaint.right)-ps.rcPaint.left,
									min(sy,ps.rcPaint.bottom)-ps.rcPaint.top,
								hdcMemory,
								pInfo->Data.Image.nScrollLeft+ps.rcPaint.left,
								pInfo->Data.Image.nScrollTop+ps.rcPaint.top,
																	SRCCOPY);
					} else {
						sx=(bm.bmWidth-pInfo->Data.Image.nScrollLeft)*
															pInfo->nZoom/100;
						if (sx==0)
							sx=1;
						sy=(bm.bmHeight-pInfo->Data.Image.nScrollTop)*
															pInfo->nZoom/100;
						if (sy==0)
							sy=1;
						if (ps.rcPaint.left<sx && ps.rcPaint.top<sy) {
							int nDstLeft,nDstTop,nDstWidth,nDstHeight;
							int nSrcLeft,nSrcTop,nSrcWidth,nSrcHeight;

							if (pInfo->nZoom%100!=0) {
								/* [Ȕ{̏ꍇAŜ`悵ȂƂ */
								nDstLeft=nDstTop=0;
								nDstWidth=sx;
								nDstHeight=sy;
								nSrcLeft=pInfo->Data.Image.nScrollLeft;
								nSrcTop=pInfo->Data.Image.nScrollTop;
								nSrcWidth=bm.bmWidth-nSrcLeft;
								nSrcHeight=bm.bmHeight-nSrcTop;
							} else {
								/* Kvȕ̂ݕ`悷悤ɂ */
								nSrcLeft=ps.rcPaint.left*100/pInfo->nZoom;
								nSrcTop=ps.rcPaint.top*100/pInfo->nZoom;
								nDstLeft=nSrcLeft*pInfo->nZoom/100;
								nDstTop=nSrcTop*pInfo->nZoom/100;
								nSrcLeft+=pInfo->Data.Image.nScrollLeft;
								nSrcTop+=pInfo->Data.Image.nScrollTop;
								nSrcWidth=
									((min(sx,ps.rcPaint.right)-nDstLeft)*100+
												(pInfo->nZoom-1))/pInfo->nZoom;
								nSrcHeight=
									((min(sy,ps.rcPaint.bottom)-nDstTop)*100+
												(pInfo->nZoom-1))/pInfo->nZoom;
								nDstWidth=nSrcWidth*pInfo->nZoom/100;
								nDstHeight=nSrcHeight*pInfo->nZoom/100;
							}
							StretchBlt(ps.hdc,nDstLeft,nDstTop,
														nDstWidth,nDstHeight,
											hdcMemory,nSrcLeft,nSrcTop,
												nSrcWidth,nSrcHeight,SRCCOPY);
						}
					}
					SelectObject(hdcMemory,hbmOld);
					DeleteDC(hdcMemory);
					if (ps.fErase) {
						if (ps.rcPaint.right>sx) {
							rc.left=max(ps.rcPaint.left,sx);
							rc.top=ps.rcPaint.top;
							rc.right=ps.rcPaint.right;
							rc.bottom=ps.rcPaint.bottom;
							FillRect(ps.hdc,&rc,hbrBackground);
						}
						if (ps.rcPaint.bottom>sy && ps.rcPaint.left<sx) {
							rc.left=ps.rcPaint.left;
							rc.top=max(ps.rcPaint.top,sy);
							rc.right=min(ps.rcPaint.right,sx);
							rc.bottom=ps.rcPaint.bottom;
							FillRect(ps.hdc,&rc,hbrBackground);
						}
					}
				}
				break;
			default:
				if (ps.fErase)
					FillRect(ps.hdc,&ps.rcPaint,hbrBackground);
			}
			DeleteObject(hbrBackground);
			EndPaint(hwnd,&ps);
		}
		return 0;

	case WM_SIZE:
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(GetParent(hwnd));

			if (pInfo->Format==PREVIEW_FORMAT_IMAGE
											&& pInfo->Data.Image.hbm!=NULL) {
				int cx=LOWORD(lParam),cy=HIWORD(lParam);
				BITMAP bm;

				GetObject(pInfo->Data.Image.hbm,sizeof(BITMAP),&bm);
				if (pInfo->fFitImageToWindow) {
					int nZoom;

					nZoom=min((cx*100)/bm.bmWidth,(cy*100)/bm.bmHeight);
					if (nZoom==0)
						nZoom=1;
					if (pInfo->nZoom!=nZoom) {
						pInfo->nZoom=nZoom;
						InvalidateRect(hwnd,NULL,FALSE);
					}
					return 0;
				}
				if (!pInfo->Data.Image.fScrollBarSetting) {
					int OldScrollLeft,OldScrollTop;
					int HScroll,VScroll;

					OldScrollLeft=pInfo->Data.Image.nScrollLeft;
					OldScrollTop=pInfo->Data.Image.nScrollTop;
					SetScrollBar(hwnd,pInfo);
					HScroll=OldScrollLeft-pInfo->Data.Image.nScrollLeft;
					VScroll=OldScrollTop-pInfo->Data.Image.nScrollTop;
					if (HScroll!=0 || VScroll!=0) {
						if (pInfo->nZoom%100==0)
							ScrollWindowEx(hwnd,HScroll*pInfo->nZoom/100,
											VScroll*pInfo->nZoom/100,
											NULL,NULL,NULL,NULL,SW_INVALIDATE);
						else
							InvalidateRect(hwnd,NULL,FALSE);
					}
				}
			} else if (pInfo->Format==PREVIEW_FORMAT_VIDEO
					&& pInfo->fFitImageToWindow && pInfo->Data.Video.fOpened) {
				RECT rc;
				int nZoom;

				MCIWndGetSource(pInfo->Data.Video.hwndMCI,&rc);
				nZoom=min((LOWORD(lParam)*100)/rc.right,
											(HIWORD(lParam)*100)/rc.bottom);
				MCIWndSetZoom(pInfo->Data.Video.hwndMCI,max(nZoom,1));
			}
		}
		return 0;

	case WM_HSCROLL:
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(GetParent(hwnd));

			if (pInfo->Format==PREVIEW_FORMAT_IMAGE
											&& pInfo->Data.Image.hbm!=NULL) {
				int nPos,nPage;
				RECT rc;
				BITMAP bm;

				nPos=pInfo->Data.Image.nScrollLeft;
				GetClientRect(hwnd,&rc);
				nPage=(rc.right*100+(pInfo->nZoom-1))/pInfo->nZoom;
				switch (LOWORD(wParam)){
				case SB_LINELEFT:
					if (pInfo->nZoom>=100)
						nPos--;
					else
						nPos-=(100+(pInfo->nZoom-1))/pInfo->nZoom;
					break;
				case SB_LINERIGHT:
					if (pInfo->nZoom>=100)
						nPos++;
					else
						nPos+=(100+(pInfo->nZoom-1))/pInfo->nZoom;
					break;
				case SB_PAGELEFT:
					nPos-=nPage;
					break;
				case SB_PAGERIGHT:
					nPos+=nPage;
					break;
				case SB_THUMBTRACK:
					nPos=HIWORD(wParam);
					break;
				default:
					return 0;
				}
				GetObject(pInfo->Data.Image.hbm,sizeof(BITMAP),&bm);
				if (nPos<0 || bm.bmWidth<=nPage)
					nPos=0;
				else if (nPos+nPage>bm.bmWidth)
					nPos=bm.bmWidth-nPage;
				if (nPos!=pInfo->Data.Image.nScrollLeft)
					ScrollImage(hwnd,pInfo,
								nPos-pInfo->Data.Image.nScrollLeft,0,TRUE);
			}
		}
		return 0;

	case WM_VSCROLL:
	case WM_MOUSEWHEEL:
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(GetParent(hwnd));

			if (pInfo->Format==PREVIEW_FORMAT_IMAGE
											&& pInfo->Data.Image.hbm!=NULL) {
				int nPos,nPage;
				RECT rc;
				BITMAP bm;

				nPos=pInfo->Data.Image.nScrollTop;
				GetClientRect(hwnd,&rc);
				nPage=(rc.bottom*100+(pInfo->nZoom-1))/pInfo->nZoom;
				GetObject(pInfo->Data.Image.hbm,sizeof(BITMAP),&bm);
				if (uMsg==WM_VSCROLL) {
					switch (LOWORD(wParam)){
					case SB_LINEUP:
						if (pInfo->nZoom>=100)
							nPos--;
						else
							nPos-=(100+(pInfo->nZoom-1))/pInfo->nZoom;
						break;
					case SB_LINEDOWN:
						if (pInfo->nZoom>=100)
							nPos++;
						else
							nPos+=(100+(pInfo->nZoom-1))/pInfo->nZoom;
						break;
					case SB_PAGEUP:
						nPos-=nPage;
						break;
					case SB_PAGEDOWN:
						nPos+=nPage;
						break;
					case SB_THUMBTRACK:
						nPos=HIWORD(wParam);
						break;
					case SB_TOP:
						nPos=0;
						break;
					case SB_BOTTOM:
						nPos=bm.bmHeight-nPage;
						break;
					default:
						return 0;
					}
				} else {
					if (pInfo->nZoom>=100)
						nPos-=(SHORT)HIWORD(wParam)/WHEEL_DELTA;
					else
						nPos-=(100+(pInfo->nZoom-1))*(SHORT)HIWORD(wParam)/
													WHEEL_DELTA/pInfo->nZoom;
				}
				if (nPos<0 || bm.bmHeight<=nPage)
					nPos=0;
				else if (nPos+nPage>bm.bmHeight)
					nPos=bm.bmHeight-nPage;
				if (nPos!=pInfo->Data.Image.nScrollTop)
					ScrollImage(hwnd,pInfo,
									0,nPos-pInfo->Data.Image.nScrollTop,TRUE);
			}
		}
		return 0;

	case WM_LBUTTONDOWN:
		SetFocus(hwnd);
		return 0;

	case WM_RBUTTONDOWN:
		{
			HWND hwndParent=GetParent(hwnd);

			SetFocus(hwnd);
			SendMessage(GetParent(hwndParent),WM_COMMAND,
				MAKEWPARAM(GetWindowLong(hwndParent,GWL_ID),PVN_RBUTTONDOWN),
														(LPARAM)hwndParent);
		}
		return 0;

	case MCIWNDM_NOTIFYMODE:
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(GetParent(hwnd));

			switch (lParam) {
			case MCI_MODE_PLAY:
				SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,CM_VIDEO_PAUSE,
																		TRUE);
				SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,CM_VIDEO_STOP,
																		TRUE);
				break;
			case MCI_MODE_PAUSE:
				SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,CM_VIDEO_PAUSE,
																		TRUE);
				SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,CM_VIDEO_STOP,
																		FALSE);
				break;
			case MCI_MODE_STOP:
				SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,CM_VIDEO_PAUSE,
																		FALSE);
				SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,CM_VIDEO_STOP,
																		TRUE);
				break;
			default:
				return 0;
			}
			SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,CM_VIDEO_PLAY,
											lParam==MCI_MODE_PLAY?TRUE:FALSE);
			SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,CM_VIDEO_PAUSE,
											lParam==MCI_MODE_PAUSE?TRUE:FALSE);
			SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,CM_VIDEO_STOP,
											lParam==MCI_MODE_STOP?TRUE:FALSE);
		}
		return 0;

	case MCIWNDM_NOTIFYPOS:
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(GetParent(hwnd));

			SendMessage(pInfo->Data.Video.hwndSeekBar,TBM_SETPOS,TRUE,lParam);
		}
		return 0;
	}
	return DefWindowProc(hwnd,uMsg,wParam,lParam);
}


/*
	摜c[o[쐬
*/
static HWND CreateImageToolbar(HWND hwndParent)
{
	static const TBBUTTON tbb[]={
		{0,	CM_IMAGE_ZOOMIN,	TBSTATE_ENABLED,	BTNS_BUTTON,	{0,0},0,0},
		{1,	CM_IMAGE_ZOOMOUT,	TBSTATE_ENABLED,	BTNS_BUTTON,	{0,0},0,0},
		{2,	CM_IMAGE_ZOOM100,	TBSTATE_ENABLED,	BTNS_BUTTON,	{0,0},0,0},
		{3,	CM_IMAGE_FITIMAGE,	TBSTATE_ENABLED,	BTNS_CHECK,		{0,0},0,0},
	/*
		{0,	0,					TBSTATE_ENABLED,	BTNS_SEP,		{0,0},0,0},
		{4,	CM_IMAGE_ROTATELEFT,	TBSTATE_ENABLED,	BTNS_BUTTON,	{0,0},	0,0},
		{5,	CM_IMAGE_ROTATERIGHT,	TBSTATE_ENABLED,	BTNS_BUTTON,	{0,0},	0,0},
	*/
	};

	return CreateToolbarEx(hwndParent,WS_CHILD | WS_VISIBLE |
				CCS_BOTTOM | CCS_NODIVIDER | TBSTYLE_TOOLTIPS | TBSTYLE_FLAT,0,
		4,hInst,IDB_IMAGEPREVIEW,tbb,numberof(tbb),0,0,0,0,sizeof(TBBUTTON));
}


/*
	c[o[쐬
*/
static HWND CreateSoundToolbar(HWND hwndParent)
{
	static const TBBUTTON tbb[]={
		{0,	CM_SOUND_PLAY,	0,	BTNS_BUTTON,	{0,0},	0,0},
		{1,	CM_SOUND_PAUSE,	0,	BTNS_BUTTON,	{0,0},	0,0},
		{2,	CM_SOUND_STOP,	0,	BTNS_BUTTON,	{0,0},	0,0},
	};

	return CreateToolbarEx(hwndParent,WS_CHILD | WS_VISIBLE |
				CCS_BOTTOM | CCS_NODIVIDER | TBSTYLE_TOOLTIPS | TBSTYLE_FLAT,0,
			3,hInst,IDB_SOUNDPLAY,tbb,numberof(tbb),0,0,0,0,sizeof(TBBUTTON));
}


/*
	c[o[쐬
*/
static HWND CreateVideoToolbar(HWND hwndParent)
{
	static const TBBUTTON tbb[]={
		{0,	CM_VIDEO_PLAY,	0,					BTNS_CHECK,		{0,0},	0,0},
		{1,	CM_VIDEO_PAUSE,	0,					BTNS_CHECK,		{0,0},	0,0},
		{2,	CM_VIDEO_STOP,	0,					BTNS_CHECK,		{0,0},	0,0},
		{0,	0,				TBSTATE_ENABLED,	BTNS_SEP,		{0,0},	0,0},
		{3,	CM_VIDEO_FIT,	TBSTATE_ENABLED,	BTNS_CHECK,		{0,0},	0,0},
	};

	return CreateToolbarEx(hwndParent,WS_CHILD | WS_VISIBLE |
				CCS_BOTTOM | CCS_NODIVIDER | TBSTYLE_TOOLTIPS | TBSTYLE_FLAT,0,
			4,hInst,IDB_VIDEOPLAY,tbb,numberof(tbb),0,0,0,0,sizeof(TBBUTTON));
}


/*
	vr[p̃f[^J
*/
static void FreePreviewData(HWND hwnd,PreviewWindowInfo *pInfo)
{
	switch (pInfo->Format) {
	case PREVIEW_FORMAT_IMAGE:
		if (pInfo->Data.Image.hbm!=NULL){
			DeleteObject(pInfo->Data.Image.hbm);
			pInfo->Data.Image.hbm=NULL;
		}
		pInfo->Data.Image.pBits=NULL;
		break;
	case PREVIEW_FORMAT_SOUND:
		if (pInfo->Data.Sound.pwfx!=NULL) {
			MemFree(pInfo->Data.Sound.pwfx);
			pInfo->Data.Sound.pwfx=NULL;
		}
		if (pInfo->Data.Sound.pData!=NULL) {
			MemFree(pInfo->Data.Sound.pData);
			pInfo->Data.Sound.pData=NULL;
		}
		if (pInfo->Data.Sound.hwo!=NULL)
			SendMessage(hwnd,WM_COMMAND,MAKEWPARAM(CM_SOUND_STOP,0),0);
		break;
	case PREVIEW_FORMAT_MIDI:
		pInfo->Data.Midi.pData=NULL;
		if (pInfo->Data.Midi.fOpened) {
			mciSendString(TEXT("stop midi"),NULL,0,NULL);
			mciSendString(TEXT("close midi"),NULL,0,NULL);
			pInfo->Data.Midi.fOpened=FALSE;
		}
		if (pInfo->Data.Midi.szTempFileName[0]!='\0') {
			DeleteFile(pInfo->Data.Midi.szTempFileName);
			pInfo->Data.Midi.szTempFileName[0]='\0';
		}
		break;
	case PREVIEW_FORMAT_MP3:
		pInfo->Data.MP3.pData=NULL;
		if (pInfo->Data.MP3.fOpened) {
			mciSendString(TEXT("stop mp3"),NULL,0,NULL);
			mciSendString(TEXT("close mp3"),NULL,0,NULL);
			pInfo->Data.MP3.fOpened=FALSE;
		}
		if (pInfo->Data.MP3.szTempFileName[0]!='\0') {
			DeleteFile(pInfo->Data.MP3.szTempFileName);
			pInfo->Data.Midi.szTempFileName[0]='\0';
		}
		break;
	case PREVIEW_FORMAT_VIDEO:
		pInfo->Data.Video.pData=NULL;
		if (pInfo->Data.Video.fOpened) {
			MCIWndClose(pInfo->Data.Video.hwndMCI);
			pInfo->Data.Video.fOpened=FALSE;
		}
		if (pInfo->Data.Video.szTempFilePath[0]!='\0') {
			DeleteFile(pInfo->Data.Video.szTempFilePath);
			pInfo->Data.Video.szTempFilePath[0]='\0';
		}
		break;
	}
}


/*
	vr[̎ނݒ肷
*/
static BOOL SetFormatType(HWND hwnd,PreviewFormatType Format)
{
	PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);
	RECT rc;

	if (pInfo->Format!=PREVIEW_FORMAT_UNDEFINED)
		FreePreviewData(hwnd,pInfo);
	if (pInfo->Format==Format)
		return TRUE;
	if (pInfo->hwndToolbar!=NULL)
		DestroyWindow(pInfo->hwndToolbar);
	switch (pInfo->Format) {
	case PREVIEW_FORMAT_IMAGE:
		ShowScrollBar(pInfo->hwndCanvas,SB_BOTH,FALSE);
		InvalidateRect(pInfo->hwndCanvas,NULL,TRUE);
		break;
	case PREVIEW_FORMAT_VIDEO:
		if (pInfo->Data.Video.hwndMCI!=NULL)
			DestroyWindow(pInfo->Data.Video.hwndMCI);
		if (pInfo->Data.Video.hwndSeekBar!=NULL)
			DestroyWindow(pInfo->Data.Video.hwndSeekBar);
		break;
	}
	switch (Format) {
	case PREVIEW_FORMAT_UNDEFINED:
		pInfo->hwndToolbar=NULL;
		break;
	case PREVIEW_FORMAT_IMAGE:
		pInfo->hwndToolbar=CreateImageToolbar(hwnd);
		if (pInfo->fFitImageToWindow)
			SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,CM_IMAGE_FITIMAGE,
																		TRUE);
		pInfo->Data.Image.hbm=NULL;
		pInfo->Data.Image.pBits=NULL;
		pInfo->Data.Image.fScrollBarSetting=FALSE;
		break;
	case PREVIEW_FORMAT_SOUND:
		pInfo->hwndToolbar=CreateSoundToolbar(hwnd);
		pInfo->Data.Sound.pwfx=NULL;
		pInfo->Data.Sound.pData=NULL;
		pInfo->Data.Sound.hwo=NULL;
		break;
	case PREVIEW_FORMAT_MIDI:
		pInfo->hwndToolbar=CreateSoundToolbar(hwnd);
		pInfo->Data.Midi.pData=NULL;
		pInfo->Data.Midi.fOpened=FALSE;
		pInfo->Data.Midi.szTempFileName[0]='\0';
		pInfo->Data.Midi.fPause=FALSE;
		break;
	case PREVIEW_FORMAT_MP3:
		pInfo->hwndToolbar=CreateSoundToolbar(hwnd);
		pInfo->Data.MP3.pData=NULL;
		pInfo->Data.MP3.fOpened=FALSE;
		pInfo->Data.MP3.szTempFileName[0]='\0';
		pInfo->Data.MP3.fPause=FALSE;
		break;
	case PREVIEW_FORMAT_VIDEO:
		pInfo->hwndToolbar=CreateVideoToolbar(hwnd);
		if (pInfo->fFitImageToWindow)
			SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,CM_VIDEO_FIT,TRUE);
		pInfo->Data.Video.pData=NULL;
		pInfo->Data.Video.fOpened=FALSE;
		pInfo->Data.Video.szTempFilePath[0]='\0';
		pInfo->Data.Video.hwndMCI=NULL;
		pInfo->Data.Video.hwndSeekBar=CreateWindow(TRACKBAR_CLASS,TEXT(""),
			WS_CHILD | WS_VISIBLE | WS_DISABLED | TBS_NOTICKS,0,0,0,20,hwnd,
															NULL,hInst,NULL);
		break;
	}
	if (pInfo->hwndToolbar!=NULL && !pInfo->fShowTipHelp)
		ShowToolbarTipHelp(pInfo->hwndToolbar,FALSE);
	pInfo->Format=Format;
	GetClientRect(hwnd,&rc);
	SendMessage(hwnd,WM_SIZE,0,MAKELPARAM(rc.right,rc.bottom));
	return TRUE;
}


/*
	EBhEvV[W
*/
static LRESULT CALLBACK PreviewWndProc(HWND hwnd,UINT uMsg,WPARAM wParam,
																LPARAM lParam)
{
	switch (uMsg) {
	case WM_CREATE:
		{
			PreviewWindowInfo *pInfo;

			pInfo=(PreviewWindowInfo*)MemAlloc(sizeof(PreviewWindowInfo));
			if (pInfo==NULL)
				return -1;
			pInfo->hwndToolbar=NULL;
			pInfo->Format=PREVIEW_FORMAT_UNDEFINED;
			pInfo->fFitImageToWindow=FALSE;
			pInfo->nZoom=100;
			pInfo->nZoomStep=20;
			pInfo->fShowTipHelp=TRUE;
			SetWindowLongPtr(hwnd,0,(LONG_PTR)pInfo);
			pInfo->hwndCanvas=CreateWindowEx(WS_EX_CLIENTEDGE,
							PREVIEW_CANVAS_WINDOW_CLASS,NULL,
							WS_CHILD | WS_VISIBLE | WS_VSCROLL | WS_HSCROLL,
							0,0,0,0,hwnd,(HMENU)0,hInst,NULL);
		}
		return 0;

	case WM_SIZE:
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);
			int nBottom=HIWORD(lParam);

			if (pInfo->hwndToolbar!=NULL){
				RECT rcToolbar;

				SendMessage(pInfo->hwndToolbar,TB_AUTOSIZE,0,0);
				GetWindowRect(pInfo->hwndToolbar,&rcToolbar);
				nBottom-=rcToolbar.bottom-rcToolbar.top;
			}
			if (pInfo->Format==PREVIEW_FORMAT_VIDEO) {
				RECT rcSeekBar;

				GetWindowRect(pInfo->Data.Video.hwndSeekBar,&rcSeekBar);
				nBottom-=rcSeekBar.bottom-rcSeekBar.top;
				MoveWindow(pInfo->Data.Video.hwndSeekBar,0,nBottom,
						LOWORD(lParam),rcSeekBar.bottom-rcSeekBar.top,TRUE);
				/* ĕ`悵ȂƃS~o */
				InvalidateRect(pInfo->Data.Video.hwndSeekBar,NULL,TRUE);
			}
			MoveWindow(pInfo->hwndCanvas,0,0,LOWORD(lParam),nBottom,TRUE);
		}
		return 0;

	case WM_NOTIFY:
		switch (((LPNMHDR)lParam)->code) {
		case TTN_NEEDTEXT:
#if 0	/* ƂƂ */
		case TTN_SHOW:
		case TTN_POP:
#endif
			return SendMessage(GetParent(hwnd),uMsg,wParam,lParam);
		}
		break;

	case PVM_SETDIB:
		/* DIBf[^ݒ肷 */
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);
			const void *pData=(const void*)lParam;
			const BITMAPINFO *pbmi;
			UINT uInfoBytes;
			struct {
				BITMAPINFOHEADER bmiHeader;
				RGBQUAD bmiColors[256];
			} bmi;

			if (((BYTE*)pData)[0]=='B' && ((BYTE*)pData)[1]=='M')
				pbmi=(BITMAPINFO*)((BYTE*)pData+sizeof(BITMAPFILEHEADER));
			else
				pbmi=(BITMAPINFO*)pData;
			SetFormatType(hwnd,PREVIEW_FORMAT_IMAGE);
			bmi.bmiHeader.biSize=sizeof(BITMAPINFOHEADER);
			bmi.bmiHeader.biWidth=pbmi->bmiHeader.biWidth;
			bmi.bmiHeader.biHeight=pbmi->bmiHeader.biHeight;
			bmi.bmiHeader.biPlanes=1;
			bmi.bmiHeader.biBitCount=pbmi->bmiHeader.biBitCount;
			bmi.bmiHeader.biCompression=BI_RGB;
			bmi.bmiHeader.biSizeImage=0;
			bmi.bmiHeader.biXPelsPerMeter=0;
			bmi.bmiHeader.biYPelsPerMeter=0;
			bmi.bmiHeader.biClrUsed=0;
			bmi.bmiHeader.biClrImportant=0;
			uInfoBytes=DIBGetInfoBytes(pData);
			if (bmi.bmiHeader.biBitCount<=8) {
				int nColors=
						(uInfoBytes-sizeof(BITMAPINFOHEADER))/sizeof(RGBQUAD);

				MemCopy(bmi.bmiColors,pbmi->bmiColors,nColors*sizeof(RGBQUAD));
				if (nColors<1<<bmi.bmiHeader.biBitCount)
					MemZero(&bmi.bmiColors[nColors],
									((1<<bmi.bmiHeader.biBitCount)-nColors)*
															sizeof(RGBQUAD));
			}
			pInfo->Data.Image.hbm=CreateDIBSection(NULL,(BITMAPINFO*)&bmi,
							DIB_RGB_COLORS,&pInfo->Data.Image.pBits,NULL,0);
			if (pInfo->Data.Image.hbm==NULL)
				return FALSE;
			pInfo->Data.Image.nScrollLeft=0;
			pInfo->Data.Image.nScrollTop=0;
			pInfo->Format=PREVIEW_FORMAT_IMAGE;
			if (!pInfo->fFitImageToWindow) {
				pInfo->nZoom=100;
			} else {
				RECT rc;

				ShowScrollBar(pInfo->hwndCanvas,SB_BOTH,FALSE);
				GetClientRect(pInfo->hwndCanvas,&rc);
				pInfo->nZoom=min(rc.right*100/pbmi->bmiHeader.biWidth,
									rc.bottom*100/pbmi->bmiHeader.biHeight);
				if (pInfo->nZoom==0)
					pInfo->nZoom=1;
			}
			if (pbmi->bmiHeader.biCompression==BI_RGB
					|| pbmi->bmiHeader.biCompression==BI_BITFIELDS)
				MemCopy(pInfo->Data.Image.pBits,(BYTE*)pbmi+uInfoBytes,
					DIB_ROW_BYTES(pbmi->bmiHeader.biWidth,
						pbmi->bmiHeader.biBitCount)*pbmi->bmiHeader.biHeight);
			else
				DIBDecodeRLE(pbmi,(BYTE*)pbmi+uInfoBytes,
						pbmi->bmiHeader.biSizeImage,pInfo->Data.Image.pBits);
			SetScrollBar(pInfo->hwndCanvas,pInfo);
			/*
			{
				RECT rc;
				GetClientRect(pInfo->hwndCanvas,&rc);
				SendMessage(pInfo->hwndCanvas,WM_SIZE,0,
											MAKELPARAM(rc.right,rc.bottom));
			}
			*/
			InvalidateRect(pInfo->hwndCanvas,NULL,TRUE);
		}
		return TRUE;

	case PVM_SETWAVE:
		/* WAVEf[^ݒ肷 */
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);
			const void *pData=(const void*)lParam;
			DWORD WaveFormatSize;
			const BYTE *pbDataChunk;

			SetFormatType(hwnd,PREVIEW_FORMAT_SOUND);
			WaveFormatSize=*(DWORD*)((BYTE*)pData+8+4+4);
			pInfo->Data.Sound.pwfx=(WAVEFORMATEX*)MemAlloc(WaveFormatSize);
			if (pInfo->Data.Sound.pwfx==NULL)
				return FALSE;
			MemCopy(pInfo->Data.Sound.pwfx,(BYTE*)pData+8+4+8,WaveFormatSize);
			pbDataChunk=(BYTE*)pData+8+4+8+WaveFormatSize;
			if (pbDataChunk[0]=='f')	/* "fact' */
				pbDataChunk+=12;
			pInfo->Data.Sound.DataSize=*(DWORD*)(pbDataChunk+4);
			pInfo->Data.Sound.pData=MemAlloc(pInfo->Data.Sound.DataSize);
			if (pInfo->Data.Sound.pData==NULL) {
				MemFree(pInfo->Data.Sound.pwfx);
				pInfo->Data.Sound.pwfx=NULL;
				return FALSE;
			}
			MemCopy(pInfo->Data.Sound.pData,pbDataChunk+8,
												pInfo->Data.Sound.DataSize);
			pInfo->Data.Sound.hwo=NULL;
			SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,CM_SOUND_PLAY,TRUE);
		}
		return TRUE;

	case PVM_SETMIDI:
		/* MIDIf[^ݒ肷 */
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

			SetFormatType(hwnd,PREVIEW_FORMAT_MIDI);
			pInfo->Data.Midi.pData=(const void*)lParam;
			pInfo->Data.Midi.DataSize=wParam;
			SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,CM_SOUND_PLAY,TRUE);
		}
		return TRUE;

	case PVM_SETMP3:
		/* MP3f[^ݒ肷 */
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

			SetFormatType(hwnd,PREVIEW_FORMAT_MP3);
			pInfo->Data.MP3.pData=(const void*)lParam;
			pInfo->Data.MP3.DataSize=wParam;
			SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,CM_SOUND_PLAY,TRUE);
		}
		return TRUE;

	case PVM_SETAVI:
		/* AVIf[^ݒ肷 */
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

			SetFormatType(hwnd,PREVIEW_FORMAT_VIDEO);
			pInfo->Data.Video.pData=(const void*)lParam;
			pInfo->Data.Video.DataSize=wParam;
			SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,CM_VIDEO_PLAY,TRUE);
		}
		return TRUE;

	case PVM_FITIMAGETOWINDOW:
		/* EBhEɍ킹 */
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

			if (pInfo->fFitImageToWindow==(wParam!=0))
				return TRUE;
			pInfo->fFitImageToWindow=wParam!=0;
			if (pInfo->Format==PREVIEW_FORMAT_IMAGE) {
				SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
								CM_IMAGE_FITIMAGE,pInfo->fFitImageToWindow);
				SendMessage(hwnd,WM_COMMAND,MAKEWPARAM(CM_IMAGE_FITIMAGE,0),0);
			} else if (pInfo->Format==PREVIEW_FORMAT_VIDEO) {
				SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
										CM_VIDEO_FIT,pInfo->fFitImageToWindow);
				SendMessage(hwnd,WM_COMMAND,MAKEWPARAM(CM_VIDEO_FIT,0),0);
			}
		}
		return TRUE;

	case PVM_RESET:
		/* eNA */
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

			if (pInfo->Format!=PREVIEW_FORMAT_UNDEFINED)
				SetFormatType(hwnd,PREVIEW_FORMAT_UNDEFINED);
		}
		return 0;

	case PVM_SHOWTIPHELP:
		/* `bvwv̗L/ݒ肷 */
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);
			BOOL fShow=wParam!=0;

			if (pInfo->fShowTipHelp!=fShow) {
				pInfo->fShowTipHelp=fShow;
				if (pInfo->hwndToolbar!=NULL)
					ShowToolbarTipHelp(pInfo->hwndToolbar,pInfo->fShowTipHelp);
			}
		}
		return TRUE;

	case PVM_GETDIBITS:
		/* DIB ̃f[^擾 */
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);
			BITMAP bm;

			if (pInfo->Format!=PREVIEW_FORMAT_IMAGE)
				return FALSE;
			GetObject(pInfo->Data.Image.hbm,sizeof(BITMAP),&bm);
			MemCopy((void*)lParam,pInfo->Data.Image.pBits,
						DIB_ROW_BYTES(bm.bmWidth,bm.bmBitsPixel)*bm.bmHeight);
		}
		return TRUE;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case CM_IMAGE_ZOOMIN:
			{
				PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

				if (pInfo->Format==PREVIEW_FORMAT_IMAGE
						&& pInfo->Data.Image.hbm!=NULL
						&& pInfo->nZoom<ZOOM_MAX) {
					if (pInfo->fFitImageToWindow) {
						pInfo->fFitImageToWindow=FALSE;
						SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
													CM_IMAGE_FITIMAGE,FALSE);
					}
					pInfo->nZoom+=pInfo->nZoomStep;
					if (pInfo->nZoom>ZOOM_MAX)
						pInfo->nZoom=ZOOM_MAX;
					SetScrollBar(pInfo->hwndCanvas,pInfo);
					InvalidateRect(pInfo->hwndCanvas,NULL,TRUE);
				}
			}
			return 0;

		case CM_IMAGE_ZOOMOUT:
			{
				PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

				if (pInfo->Format==PREVIEW_FORMAT_IMAGE
						&& pInfo->Data.Image.hbm!=NULL
						&& pInfo->nZoom>ZOOM_MIN) {
					if (pInfo->fFitImageToWindow) {
						pInfo->fFitImageToWindow=FALSE;
						SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
													CM_IMAGE_FITIMAGE,FALSE);
					}
					pInfo->nZoom-=pInfo->nZoomStep;
					if (pInfo->nZoom<ZOOM_MIN)
						pInfo->nZoom=ZOOM_MIN;
					SetScrollBar(pInfo->hwndCanvas,pInfo);
					InvalidateRect(pInfo->hwndCanvas,NULL,TRUE);
				}
			}
			return 0;

		case CM_IMAGE_ZOOM100:
			{
				PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

				if (pInfo->Format==PREVIEW_FORMAT_IMAGE
						&& pInfo->Data.Image.hbm!=NULL && pInfo->nZoom!=100) {
					if (pInfo->fFitImageToWindow) {
						pInfo->fFitImageToWindow=FALSE;
						SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
													CM_IMAGE_FITIMAGE,FALSE);
					}
					pInfo->nZoom=100;
					SetScrollBar(pInfo->hwndCanvas,pInfo);
					InvalidateRect(pInfo->hwndCanvas,NULL,TRUE);
				}
			}
			return 0;

		case CM_IMAGE_FITIMAGE:
			{
				PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

				pInfo->fFitImageToWindow=!pInfo->fFitImageToWindow;
				if (pInfo->fFitImageToWindow
						&& pInfo->Format==PREVIEW_FORMAT_IMAGE
						&& pInfo->Data.Image.hbm!=NULL) {
					RECT rc;
					BITMAP bm;
					int nZoom;

					ShowScrollBar(pInfo->hwndCanvas,SB_BOTH,FALSE);
					GetClientRect(pInfo->hwndCanvas,&rc);
					GetObject(pInfo->Data.Image.hbm,sizeof(BITMAP),&bm);
					nZoom=min((rc.right*100)/bm.bmWidth,
												(rc.bottom*100)/bm.bmHeight);
					if (nZoom==0)
						nZoom=1;
					if (pInfo->nZoom!=nZoom) {
						pInfo->nZoom=nZoom;
						pInfo->Data.Image.nScrollLeft=0;
						pInfo->Data.Image.nScrollTop=0;
						InvalidateRect(pInfo->hwndCanvas,NULL,TRUE);
					}
				}
			}
			return 0;

		case CM_SOUND_PLAY:
			/* ̍Đ */
			{
				PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

				if (pInfo->Format==PREVIEW_FORMAT_SOUND) {
					HWAVEOUT hwo;

					if (pInfo->Data.Sound.pwfx==NULL
											|| pInfo->Data.Sound.pData==NULL)
						return 0;
					if (pInfo->Data.Sound.hwo!=NULL) {
						if (pInfo->Data.Sound.fPause)
							SendMessage(hwnd,WM_COMMAND,
											MAKEWPARAM(CM_SOUND_PAUSE,0),0);
						return 0;
					}
					if (waveOutOpen(&hwo,WAVE_MAPPER,pInfo->Data.Sound.pwfx,0,0,
															WAVE_FORMAT_QUERY))
						return FALSE;
					if (waveOutOpen(&hwo,WAVE_MAPPER,pInfo->Data.Sound.pwfx,
												(UINT)hwnd,0,CALLBACK_WINDOW))
						return FALSE;
					MemZero(&pInfo->Data.Sound.wh,sizeof(WAVEHDR));
					pInfo->Data.Sound.wh.lpData=(LPSTR)pInfo->Data.Sound.pData;
					pInfo->Data.Sound.wh.dwBufferLength=
													pInfo->Data.Sound.DataSize;
					pInfo->Data.Sound.wh.dwFlags=WHDR_DONE;
					pInfo->Data.Sound.wh.dwLoops=0;
					if (waveOutPrepareHeader(hwo,&pInfo->Data.Sound.wh,
															sizeof(WAVEHDR))) {
						waveOutClose(hwo);
						return FALSE;
					}
					if (waveOutWrite(hwo,&pInfo->Data.Sound.wh,
														sizeof(WAVEHDR))!=0) {
						waveOutUnprepareHeader(hwo,&pInfo->Data.Sound.wh,
															sizeof(WAVEHDR));
						waveOutClose(hwo);
						return FALSE;
					}
					pInfo->Data.Sound.hwo=hwo;
					pInfo->Data.Sound.fPause=FALSE;
				} else if (pInfo->Format==PREVIEW_FORMAT_MIDI) {
					if (pInfo->Data.Midi.szTempFileName[0]=='\0') {
						TCHAR szTempDirectory[MAX_PATH];
						TCHAR szFileName[MAX_PATH],szFilePath[MAX_PATH];
						int i;

						GetTempPath(numberof(szTempDirectory),szTempDirectory);
						for (i=0;i<1000;i++) {
							wsprintf(szFileName,TEXT("DCTMP%03d.mid"),i);
							MergeFileName(szFilePath,szTempDirectory,
																szFileName);
							if (!FileExists(szFilePath))
								break;
						}
						if (i==1000)
							return 0;
						if (MIDISaveToFile(szFilePath,pInfo->Data.Midi.pData,
									pInfo->Data.Midi.DataSize)!=ERR_SUCCESS)
							return 0;
						lstrcpy(pInfo->Data.Midi.szTempFileName,szFilePath);
					}
					if (!pInfo->Data.Midi.fOpened) {
						TCHAR szText[MAX_PATH+64];

						wsprintf(szText,
								TEXT("open \"%s\" type sequencer alias midi"),
											pInfo->Data.Midi.szTempFileName);
						if (mciSendString(szText,NULL,0,NULL)!=0)
							return 0;
						pInfo->Data.Midi.fOpened=TRUE;
					}
					if (pInfo->Data.Midi.fPause) {
						SendMessage(hwnd,WM_COMMAND,
											MAKEWPARAM(CM_SOUND_PAUSE,0),0);
						return 0;
					}
					mciSendString(TEXT("seek midi to start"),NULL,0,NULL);
					mciSendString(TEXT("play midi notify"),NULL,0,hwnd);
				} else if (pInfo->Format==PREVIEW_FORMAT_MP3) {
					if (pInfo->Data.MP3.szTempFileName[0]=='\0') {
						TCHAR szTempDirectory[MAX_PATH];
						TCHAR szFileName[MAX_PATH],szFilePath[MAX_PATH];
						int i;

						GetTempPath(numberof(szTempDirectory),szTempDirectory);
						for (i=0;i<1000;i++) {
							wsprintf(szFileName,TEXT("DCTMP%03d.mp3"),i);
							MergeFileName(szFilePath,szTempDirectory,
																szFileName);
							if (!FileExists(szFilePath))
								break;
						}
						if (i==1000)
							return 0;
						if (MP3SaveToFile(szFilePath,pInfo->Data.MP3.pData,
									pInfo->Data.MP3.DataSize)!=ERR_SUCCESS)
							return 0;
						lstrcpy(pInfo->Data.Midi.szTempFileName,szFilePath);
					}
					if (!pInfo->Data.MP3.fOpened) {
						TCHAR szText[MAX_PATH+64];

						wsprintf(szText,
								TEXT("open \"%s\" alias mp3"),
											pInfo->Data.Midi.szTempFileName);
						if (mciSendString(szText,NULL,0,NULL)!=0)
							return 0;
						pInfo->Data.MP3.fOpened=TRUE;
					}
					if (pInfo->Data.MP3.fPause) {
						SendMessage(hwnd,WM_COMMAND,
											MAKEWPARAM(CM_SOUND_PAUSE,0),0);
						return 0;
					}
					mciSendString(TEXT("seek mp3 to start"),NULL,0,NULL);
					mciSendString(TEXT("play mp3 notify"),NULL,0,hwnd);
				} else
					return 0;
				SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,CM_SOUND_PLAY,
																		TRUE);
				SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,CM_SOUND_PAUSE,
																		TRUE);
				SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,CM_SOUND_STOP,
																		TRUE);
			}
			return 0;

		case CM_SOUND_PAUSE:
			/* ̈ꎞ~ */
			{
				PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

				if (pInfo->Format==PREVIEW_FORMAT_SOUND) {
					if (pInfo->Data.Sound.hwo!=NULL) {
						if (pInfo->Data.Sound.fPause) {
							waveOutRestart(pInfo->Data.Sound.hwo);
							pInfo->Data.Sound.fPause=FALSE;
						} else {
							waveOutPause(pInfo->Data.Sound.hwo);
							pInfo->Data.Sound.fPause=TRUE;
						}
						SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
									CM_SOUND_PAUSE,pInfo->Data.Sound.fPause);
						SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
									CM_SOUND_PLAY,!pInfo->Data.Sound.fPause);
					}
				} else if (pInfo->Format==PREVIEW_FORMAT_MIDI) {
					if (pInfo->Data.Midi.fOpened) {
						if (pInfo->Data.Midi.fPause)
							mciSendString(TEXT("play midi notify"),NULL,0,
																		hwnd);
						else
							mciSendString(TEXT("pause midi"),NULL,0,NULL);
						pInfo->Data.Midi.fPause=!pInfo->Data.Midi.fPause;
						SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
									CM_SOUND_PAUSE,pInfo->Data.Midi.fPause);
						SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
									CM_SOUND_PLAY,!pInfo->Data.Midi.fPause);
					}
				} else if (pInfo->Format==PREVIEW_FORMAT_MP3) {
					if (pInfo->Data.MP3.fOpened) {
						if (pInfo->Data.MP3.fPause)
							mciSendString(TEXT("play mp3 notify"),NULL,0,
																		hwnd);
						else
							mciSendString(TEXT("pause mp3"),NULL,0,NULL);
						pInfo->Data.MP3.fPause=!pInfo->Data.MP3.fPause;
						SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
									CM_SOUND_PAUSE,pInfo->Data.MP3.fPause);
						SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
									CM_SOUND_PLAY,!pInfo->Data.MP3.fPause);
					}
				}
			}
			return 0;

		case CM_SOUND_STOP:
			/* ̒~ */
			{
				PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

				if (pInfo->Format==PREVIEW_FORMAT_SOUND) {
					if (pInfo->Data.Sound.hwo!=NULL) {
						waveOutReset(pInfo->Data.Sound.hwo);
						waveOutUnprepareHeader(pInfo->Data.Sound.hwo,
										&pInfo->Data.Sound.wh,sizeof(WAVEHDR));
						waveOutClose(pInfo->Data.Sound.hwo);
						pInfo->Data.Sound.hwo=NULL;
						if (pInfo->Data.Sound.fPause) {
							SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
														CM_SOUND_PAUSE,FALSE);
							pInfo->Data.Sound.fPause=FALSE;
						}
					}
				} else if (pInfo->Format==PREVIEW_FORMAT_MIDI) {
					if (pInfo->Data.Midi.fOpened) {
						mciSendString(TEXT("stop midi"),NULL,0,NULL);
						if (pInfo->Data.Midi.fPause) {
							SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
														CM_SOUND_PAUSE,FALSE);
							pInfo->Data.Midi.fPause=FALSE;
						}
					}
				} else if (pInfo->Format==PREVIEW_FORMAT_MP3) {
					if (pInfo->Data.MP3.fOpened) {
						mciSendString(TEXT("stop mp3"),NULL,0,NULL);
						if (pInfo->Data.MP3.fPause) {
							SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
														CM_SOUND_PAUSE,FALSE);
							pInfo->Data.MP3.fPause=FALSE;
						}
					}
				}
				SendMessage(pInfo->hwndToolbar,TB_CHECKBUTTON,
														CM_SOUND_PLAY,FALSE);
				SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,
														CM_SOUND_PAUSE,FALSE);
				SendMessage(pInfo->hwndToolbar,TB_ENABLEBUTTON,
														CM_SOUND_STOP,FALSE);
			}
			return 0;

		case CM_VIDEO_PLAY:
			/* ̍Đ */
			{
				PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

				if (pInfo->Data.Video.szTempFilePath[0]=='\0') {
					TCHAR szTempDirectory[MAX_PATH];
					TCHAR szFileName[MAX_PATH],szFilePath[MAX_PATH];
					int i;

					GetTempPath(numberof(szTempDirectory),szTempDirectory);
					for (i=0;i<1000;i++) {
						wsprintf(szFileName,TEXT("DCTMP%03d.avi"),i);
						MergeFileName(szFilePath,szTempDirectory,szFileName);
						if (!FileExists(szFilePath))
							break;
					}
					if (i==1000)
						return 0;
					if (AVISaveToFile(szFilePath,pInfo->Data.Video.pData,
									pInfo->Data.Video.DataSize)!=ERR_SUCCESS)
						return 0;
					lstrcpy(pInfo->Data.Video.szTempFilePath,szFilePath);
				}
				if (pInfo->Data.Video.hwndMCI==NULL) {
					pInfo->Data.Video.hwndMCI=MCIWndCreate(pInfo->hwndCanvas,
							hInst,WS_CHILD | WS_VISIBLE | MCIWNDF_NOMENU |
								MCIWNDF_NOPLAYBAR | MCIWNDF_NOTIFYMODE |
								MCIWNDF_NOTIFYPOS | MCIWNDF_NOTIFYMEDIA,NULL);
					if (pInfo->Data.Video.hwndMCI==NULL)
						return 0;
					MCIWndSetTimers(pInfo->Data.Video.hwndMCI,100,200);
				}
				if (!pInfo->Data.Video.fOpened) {
					int nStart;

					if (MCIWndOpen(pInfo->Data.Video.hwndMCI,
										pInfo->Data.Video.szTempFilePath,NULL))
						return 0;
					pInfo->Data.Video.fOpened=TRUE;
					if (MCIWndGetTimeFormat(pInfo->Data.Video.hwndMCI,NULL,0)!=
													MCI_FORMAT_MILLISECONDS)
						MCIWndUseTime(pInfo->Data.Video.hwndMCI);
					EnableWindow(pInfo->Data.Video.hwndSeekBar,TRUE);
					nStart=MCIWndGetStart(pInfo->Data.Video.hwndMCI);
					SendMessage(pInfo->Data.Video.hwndSeekBar,TBM_SETRANGEMIN,
																FALSE,nStart);
					SendMessage(pInfo->Data.Video.hwndSeekBar,TBM_SETRANGEMAX,
								TRUE,MCIWndGetEnd(pInfo->Data.Video.hwndMCI));
					SendMessage(pInfo->Data.Video.hwndSeekBar,TBM_SETPOS,TRUE,
																	nStart);
					if (pInfo->fFitImageToWindow) {
						RECT rc;

						GetClientRect(pInfo->hwndCanvas,&rc);
						SendMessage(pInfo->hwndCanvas,WM_SIZE,0,
											MAKELPARAM(rc.right,rc.bottom));
					}
				}
				if (MCIWndGetMode(pInfo->Data.Video.hwndMCI,NULL,0)==
																MCI_MODE_PAUSE)
					MCIWndResume(pInfo->Data.Video.hwndMCI);
				else
					MCIWndPlay(pInfo->Data.Video.hwndMCI);
			}
			return 0;

		case CM_VIDEO_PAUSE:
			/* ̈ꎞ~ */
			{
				PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

				if (MCIWndGetMode(pInfo->Data.Video.hwndMCI,NULL,0)==
																MCI_MODE_PAUSE)
					MCIWndResume(pInfo->Data.Video.hwndMCI);
				else
					MCIWndPause(pInfo->Data.Video.hwndMCI);
			}
			return 0;

		case CM_VIDEO_STOP:
			/* ̒~ */
			{
				PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

				MCIWndStop(pInfo->Data.Video.hwndMCI);
				MCIWndHome(pInfo->Data.Video.hwndMCI);
			}
			return 0;

		case CM_VIDEO_FIT:
			/* EBhEɍ킹 */
			{
				PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

				pInfo->fFitImageToWindow=!pInfo->fFitImageToWindow;
				if (pInfo->Data.Video.fOpened) {
					if (pInfo->fFitImageToWindow) {
						RECT rc;

						GetClientRect(pInfo->hwndCanvas,&rc);
						SendMessage(pInfo->hwndCanvas,WM_SIZE,0,
											MAKELPARAM(rc.right,rc.bottom));
						/*
						RECT rcCanvas,rcSource;
						BITMAP bm;
						int nZoom;

						GetClientRect(pInfo->hwndCanvas,&rcCanvas);
						MCIWndGetSource(pInfo->Data.Video.hwndMCI,&rcSource);
						nZoom=min((rcCanvas.right*100)/rcSource.right,
										(rcCanvas.bottom*100)/rcSource.bottom);
						MCIWndSetZoom(pInfo->Data.Video.hwndMCI,max(nZoom,1));
						*/
					} else
						MCIWndSetZoom(pInfo->Data.Video.hwndMCI,100);
				}
			}
			return 0;
		}
		return 0;

	case MM_WOM_DONE:
		SendMessage(hwnd,WM_COMMAND,MAKEWPARAM(CM_SOUND_STOP,0),0);
		return 0;

	case MM_MCINOTIFY:
		if (wParam==MCI_NOTIFY_SUCCESSFUL)
			SendMessage(hwnd,WM_COMMAND,MAKEWPARAM(CM_SOUND_STOP,0),0);
		return 0;

	case WM_HSCROLL:
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);
			BOOL fPlaying;

			fPlaying=MCIWndGetMode(pInfo->Data.Video.hwndMCI,NULL,0)==
																MCI_MODE_PLAY;
			MCIWndSeek(pInfo->Data.Video.hwndMCI,
					SendMessage(pInfo->Data.Video.hwndSeekBar,TBM_GETPOS,0,0));
			if (fPlaying)
				MCIWndPlay(pInfo->Data.Video.hwndMCI);
		}
		return 0;

	case WM_DESTROY:
		{
			PreviewWindowInfo *pInfo=GetPreviewWindowInfo(hwnd);

			if (pInfo->Format!=PREVIEW_FORMAT_UNDEFINED)
				FreePreviewData(hwnd,pInfo);
			MemFree(pInfo);
			SetWindowLongPtr(hwnd,0,(LONG_PTR)(PreviewWindowInfo*)NULL);
		}
		return 0;
	}
	return DefWindowProc(hwnd,uMsg,wParam,lParam);
}


/*
	vr[EBhENX̓o^
*/
BOOL RegisterPreviewWindowClass(HINSTANCE hinst)
{
	WNDCLASS wc;

	/* vr[EBhE */
	wc.style=0;
	wc.lpfnWndProc=PreviewWndProc;
	wc.cbClsExtra=0;
	wc.cbWndExtra=sizeof(PreviewWindowInfo*);
	wc.hInstance=hinst;
	wc.hIcon=NULL;
	wc.hCursor=LoadCursor(NULL,IDC_ARROW);
	wc.hbrBackground=(HBRUSH)(COLOR_3DFACE+1);
	wc.lpszMenuName=NULL;
	wc.lpszClassName=PREVIEW_WINDOW_CLASS;
	if (RegisterClass(&wc)==0)
		return FALSE;
	/* LoX */
	wc.style=0;
	wc.lpfnWndProc=PreviewCanvasWndProc;
	wc.cbClsExtra=0;
	wc.cbWndExtra=0;
	wc.hInstance=hinst;
	wc.hIcon=NULL;
	wc.hCursor=LoadCursor(NULL,IDC_ARROW);
	wc.hbrBackground=NULL;
	wc.lpszMenuName=NULL;
	wc.lpszClassName=PREVIEW_CANVAS_WINDOW_CLASS;
	return RegisterClass(&wc)!=0;
}
